<?php
session_start();
require '../config.php';
require '../functions.php';

// Verificar si la caja está aperturada
if (!isCajaAperturada($pdo)) {
    header('Location: ../caja_no_aperturada.php');
    exit;
}

// Verifica si el usuario está autenticado
if (!isset($_SESSION['usuario_id'])) {
    header('Location: ../index.php');
    exit;
}
// Verifica si el usuario tiene permiso para acceder a la página
if (!usuarioTienePermiso($pdo, $_SESSION['usuario_id'], 'ver_rentas')) {
    header('Location: ../sin_permiso.php');
    exit;
}

// Verifica los permisos del usuario para ver, editar y eliminar
$usuario_id = $_SESSION['usuario_id'];
$permisos = [
    'ver' => usuarioTienePermiso($pdo, $usuario_id, 'ver_rentas'),
    'editar' => usuarioTienePermiso($pdo, $usuario_id, 'editar_rentas'),
    'eliminar' => usuarioTienePermiso($pdo, $usuario_id, 'borrar_rentas'),
];


// Número de registros por página
$registrosPorPagina = 10;

// Obtener el número de página actual
$paginaActual = isset($_GET['pagina']) ? (int)$_GET['pagina'] : 1;
$paginaActual = $paginaActual > 0 ? $paginaActual : 1;

// Calcular el desplazamiento
$inicio = ($paginaActual - 1) * $registrosPorPagina;

// Función para obtener las rentas con paginación
function getRentas($pdo, $inicio, $registrosPorPagina, $search = '') {
    if ($search === 'fecha_devolucion_hoy') {
        $fechaActual = date('Y-m-d');
        $sql = "
        SELECT rentas.id_renta, clientes.nombre AS nombre_cliente, clientes.apellido AS apellido_cliente, 
               CONCAT(habitaciones.nombre_habitacion, ' ', habitaciones.numero_habitacion	, ' ', habitaciones.tipo) AS auto, usuarios.nombre_usuario, 
               rentas.fecha_renta, rentas.fecha_devolucion, rentas.precio_total, rentas.estado, rentas.pago,
               DATEDIFF(rentas.fecha_devolucion, rentas.fecha_renta) AS dias_renta
        FROM rentas
        JOIN clientes ON rentas.id_cliente = clientes.id_cliente
        JOIN habitaciones ON rentas.id_auto = habitaciones.id_habitacion
        JOIN usuarios ON rentas.id_usuario = usuarios.id_usuario
        WHERE rentas.fecha_devolucion = :fecha_actual
        ORDER BY rentas.id_renta DESC
        LIMIT :inicio, :registrosPorPagina";
        $stmt = $pdo->prepare($sql);
        $stmt->bindValue(':fecha_actual', $fechaActual);
    } else {
        $sql = "
        SELECT rentas.id_renta, clientes.nombre AS nombre_cliente, clientes.apellido AS apellido_cliente, 
               CONCAT(habitaciones.nombre_habitacion, ' ', habitaciones.numero_habitacion	, ' ', habitaciones.tipo) AS auto, usuarios.nombre_usuario, 
               rentas.fecha_renta, rentas.fecha_devolucion, rentas.precio_total, rentas.estado, rentas.pago,
               DATEDIFF(rentas.fecha_devolucion, rentas.fecha_renta) AS dias_renta
        FROM rentas
        JOIN clientes ON rentas.id_cliente = clientes.id_cliente
        JOIN habitaciones ON rentas.id_auto = habitaciones.id_habitacion
        JOIN usuarios ON rentas.id_usuario = usuarios.id_usuario
        WHERE CONCAT(clientes.nombre, ' ', clientes.apellido, ' ', habitaciones.nombre_habitacion, ' ', habitaciones.numero_habitacion	, ' ', usuarios.nombre_usuario, ' ', rentas.fecha_renta, ' ', rentas.fecha_devolucion, ' ', rentas.precio_total, ' ', rentas.estado) LIKE :search
        ORDER BY rentas.id_renta DESC
        LIMIT :inicio, :registrosPorPagina";
        $stmt = $pdo->prepare($sql);
        $stmt->bindValue(':search', "%$search%");
    }
    $stmt->bindValue(':inicio', (int)$inicio, PDO::PARAM_INT);
    $stmt->bindValue(':registrosPorPagina', (int)$registrosPorPagina, PDO::PARAM_INT);
    $stmt->execute();
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}


// Función para obtener el número total de rentas
function getTotalRentas($pdo, $search = '') {
    if ($search === 'fecha_devolucion_hoy') {
        $fechaActual = date('Y-m-d');
        $sql = "
        SELECT COUNT(*) 
        FROM rentas
        JOIN clientes ON rentas.id_cliente = clientes.id_cliente
        JOIN habitaciones ON rentas.id_auto = habitaciones.id_habitacion
        JOIN usuarios ON rentas.id_usuario = usuarios.id_usuario
        WHERE rentas.fecha_devolucion = :fecha_actual";
        $stmt = $pdo->prepare($sql);
        $stmt->bindValue(':fecha_actual', $fechaActual);
    } else {
        $sql = "
        SELECT COUNT(*) 
        FROM rentas
        JOIN clientes ON rentas.id_cliente = clientes.id_cliente
        JOIN habitaciones ON rentas.id_auto = habitaciones.id_habitacion
        JOIN usuarios ON rentas.id_usuario = usuarios.id_usuario
        WHERE CONCAT(clientes.nombre, ' ', clientes.apellido, ' ', habitaciones.nombre_habitacion, ' ', habitaciones.numero_habitacion	, ' ', usuarios.nombre_usuario, ' ', rentas.fecha_renta, ' ', rentas.fecha_devolucion, ' ', rentas.precio_total, ' ', rentas.estado) LIKE :search";
        $stmt = $pdo->prepare($sql);
        $stmt->bindValue(':search', "%$search%");
    }
    $stmt->execute();
    return $stmt->fetchColumn();
}


// Buscar rentas
$search = isset($_POST['search']) ? $_POST['search'] : '';
$totalRentas = getTotalRentas($pdo, $search);
$totalPaginas = ceil($totalRentas / $registrosPorPagina);
$rentas = getRentas($pdo, $inicio, $registrosPorPagina, $search);

// Obtener la fecha actual
$fechaActual = date('Y-m-d');
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Reservaciones - Sistema Hotel</title>
    <link href="https://maxcdn.bootstrapcdn.com/bootstrap/4.5.2/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css" rel="stylesheet">
    <!-- SweetAlert2 CSS -->
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/sweetalert2@11/dist/sweetalert2.min.css">
    <style>
         .module-header {
            margin-top: 30px;
            margin-bottom: 15px;
        }
        .btn-primary {
            border-radius: 50px;
            font-size: 0.875rem;
        }

        .input-group {
            width: 100%;
            max-width: 500px;
        }

        .form-control {
            border-radius: 50px 0 0 50px;
            box-shadow: none;
        }

        .input-group-append .btn-secondary {
            border-radius: 0 50px 50px 0;
        }

        .btn-info {
            border-radius: 50px;
            font-size: 0.875rem;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
        }
        #message {
            position: fixed;
            top: 10px;
            right: 10px;
            width: 300px;
            z-index: 1050; /* Asegúrate de que esté encima de otros elementos */
        }
       .modal-content {
            padding: 20px;
        }
        .search-results {
            border: 1px solid #ddd;
            max-height: 200px;
            overflow-y: auto;
            position: absolute;
            width: 100%;
            background-color: white;
            z-index: 1000;
        }
        .search-results div {
            padding: 8px;
            cursor: pointer;
        }
        .search-results div:hover {
            background-color: #f1f1f1;
        }
        highlight {
            background-color: #ffcccc; /* Rojo suave */
        }
    </style>
</head>
<body>
    <?php include '../menu.php'; ?>

    <div class="main-content">
        <div class="d-flex align-items-center justify-content-center p-3 my-3 text-white bg-dark rounded shadow-sm">
    <div class="lh-1 text-center">
        <h2 class="mb-4 text-center">
            <i class="fas fa-calendar-alt"></i> <!-- Icono de FontAwesome -->
            Reservaciones
        </h2>
    </div>
</div>

           <!-- Contenedor para el mensaje cuando no se encuentran rentas -->
<div id="noResultsMessage" class="alert alert-warning d-none" role="alert">
    No se encontró ninguna renta.
</div>

<!-- Contenedor flex para el botón y el formulario de búsqueda -->
<div class="d-flex flex-wrap align-items-center mb-3">
    <!-- Botón para agregar una nueva renta -->
    <button class="btn btn-primary mr-3 mb-2" data-toggle="modal" data-target="#addModal">
        <i class="fas fa-plus"></i> Agregar Reserva
    </button>

    <!-- Cuadro de búsqueda con clase para ancho personalizado -->
    <div class="input-group mr-2 mb-2" style="width: 400px;">
        <input type="text" class="form-control" id="searchInput" placeholder="Buscar reserva... Cliente, habitación y estado" value="<?php echo htmlspecialchars($search); ?>">
        <div class="input-group-append">
            <button class="btn btn-secondary" type="button" id="searchButton">
                <i class="fas fa-search"></i>
            </button>
        </div>
    </div>

    <!-- Campo para filtrar por fecha de devolución -->
    <div class="d-flex align-items-center mb-2" style="max-width: 350px;">
        <label for="fechaDevolucionInput" class="mr-2 mb-0">Fecha de check-out:</label>
        <div class="input-group">
            <input type="date" class="form-control" id="fechaDevolucionInput">
            <div class="input-group-append">
                <button class="btn btn-secondary" type="button" id="filterButton">
                    <i class="fas fa-filter"></i> Filtrar
                </button>
            </div>
        </div>
    </div>
    <div class="form-group col-md-2 mb-2 d-flex align-items-end">
        <a id="generatePdf" class="btn btn-info" href="crud_reporte.php">
            <i class="fas fa-file-pdf"></i> Reporte
        </a>
    </div>
    <div class="form-group col-md-3 mb-2 d-flex align-items-end">
    <button id="enviarRecordatorio" class="btn btn-success">
    <i class="fas fa-envelope"></i> Recordatorio de check-out
    </button>
    </div>
</div>




            <!-- Tabla de rentas -->
            <table class="table table-bordered table-striped">
                <thead class="thead-dark">
                    <tr>
                        <th>ID</th>
                        <th>Cliente</th>
                        <th>Habitación</th>
                        <th>Fecha/check-in</th>
                        <th>Fecha/check-out</th>
                        <th>Estadía</th>
                        <th>Total</th>
                        <th>Estado</th>
                        <th>Pago</th>
                        <th>Usuario</th>
                        <th>Acciones</th>
                    </tr>
                </thead>
                <tbody id="rentaTableBody">
                    <?php foreach ($rentas as $renta): ?>
                        <?php 
                            // Verificar si la fecha de devolución es igual a la fecha actual
                            $highlightClass = ($renta['fecha_devolucion'] == $fechaActual) ? 'highlight' : '';
                        ?>
                        <tr class="<?php echo $highlightClass; ?>">
                            <td><?php echo htmlspecialchars($renta['id_renta']); ?></td>
                            <td><?php echo htmlspecialchars($renta['nombre_cliente'].' '.$renta['apellido_cliente']); ?></td>
                            <td><?php echo htmlspecialchars($renta['auto']); ?></td>
                            <td><?php echo htmlspecialchars(formatearFecha($renta['fecha_renta'])); ?></td>
                            <td><?php echo htmlspecialchars(formatearFecha($renta['fecha_devolucion'])); ?></td>
                            <td><span class="badge badge-primary"><?php echo htmlspecialchars($renta['dias_renta']); ?></span></td>
                            <td><?php echo htmlspecialchars(formatMonto($moneda, $renta['precio_total'])); ?></td>
                            <td><?php echo htmlspecialchars($renta['estado']); ?></td>
                            <td>
                                <?php 
                                    // Asignar la clase de badge según el estado del pago
                                    $badgeClass = $renta['pago'] === 'Pendiente' ? 'badge-danger' : 'badge-success';
                                    ?>
                                    <span class="badge <?php echo htmlspecialchars($badgeClass); ?>">
                                        <?php echo htmlspecialchars($renta['pago']); ?>
                                    </span>
                                </td>
                                <td><?php echo htmlspecialchars($renta['nombre_usuario']); ?></td>
                                <td>
                                <div class="btn-group" role="group">
                                    <?php if ($permisos['editar']) : ?>
                                        <button class="btn btn-warning btn-sm edit-btn" data-id="<?php echo $renta['id_renta']; ?>" data-toggle="modal" data-target="#editModal" title="Editar renta" data-toggle="tooltip">
                                            <i class="fas fa-edit"></i>
                                        </button>
                                    <?php endif; ?>
                                    
                                    <?php if ($permisos['eliminar']) : ?>
                                        <button class="btn btn-danger btn-sm delete-btn" data-id="<?php echo $renta['id_renta']; ?>" title="Eliminar renta" data-toggle="tooltip">
                                            <i class="fas fa-trash-alt"></i>
                                        </button>
                                    <?php endif; ?>
                                    
                                    <?php if ($permisos['ver']) : ?>
                                        <!--<button class="btn btn-info btn-sm print-btn" onclick="window.open('../reportes/print_contract.php?id_renta=<?php echo $renta['id_renta']; ?>', '_blank')" title="Imprimir contrato" data-toggle="tooltip">
                                            <i class="fas fa-print"></i>
                                        </button>-->
                                        <a class="btn btn-info btn-sm up-btn mr-2" href="../cuenta/crud_cuenta.php?id_cuenta=<?php echo $renta['id_renta']; ?>" title="Ver Cuenta" data-toggle="tooltip">
                                        <i class="fa fa-address-card"></i>
                                    </a>
                                    <?php endif; ?>
                                    
                                    <?php if ($permisos['editar']) : ?>
                                       <!-- <button class="btn btn-primary btn-sm details-btn" data-id="<?php echo $renta['id_renta']; ?>" data-toggle="modal" data-target="#detailsModal" title="Ver detalles" data-toggle="tooltip">
                                            <i class="fas fa-info-circle"></i>
                                        </button>-->
                                    <?php endif; ?>
                                </div>

                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>


            <!-- Paginación -->
            <nav aria-label="Page navigation">
                <ul class="pagination">
                    <li class="page-item <?php echo $paginaActual <= 1 ? 'disabled' : ''; ?>">
                        <a class="page-link" href="?pagina=<?php echo max(1, $paginaActual - 1); ?>" aria-label="Previous">
                            <span aria-hidden="true">&laquo;</span>
                        </a>
                    </li>
                    <?php for ($i = 1; $i <= $totalPaginas; $i++): ?>
                        <li class="page-item <?php echo $i == $paginaActual ? 'active' : ''; ?>">
                            <a class="page-link" href="?pagina=<?php echo $i; ?>"><?php echo $i; ?></a>
                        </li>
                    <?php endfor; ?>
                    <li class="page-item <?php echo $paginaActual >= $totalPaginas ? 'disabled' : ''; ?>">
                        <a class="page-link" href="?pagina=<?php echo min($totalPaginas, $paginaActual + 1); ?>" aria-label="Next">
                            <span aria-hidden="true">&raquo;</span>
                        </a>
                    </li>
                </ul>
            </nav>

        <!-- Contenedor para mensajes -->
        <div id="message" class="alert d-none"></div>

<!-- Modal para agregar una nueva renta -->
<div class="modal fade" id="addModal" tabindex="-1" role="dialog" aria-labelledby="addModalLabel" aria-hidden="true">
    <div class="modal-dialog" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="addModalLabel">Agregar Nueva Reservacción</h5>
                <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
            <div class="modal-body">
                <form id="addForm">
                <div class="form-group">
                                        <label for="addClienteInput">Cliente</label>
                                        <div class="input-group">
                                            <input type="text" class="form-control" id="addClienteInput" placeholder="Buscar cliente..." required autocomplete="off">
                                            <button type="button" class="btn btn-primary" id="addClienteButton">
                                                <i class="fas fa-plus"></i> Nuevo
                                            </button>
                                        </div>
                                        <div id="addClienteResults" class="search-results"></div>
                                        <input type="hidden" id="addClienteId" name="id_cliente"> <!-- Campo oculto para el ID del cliente -->
                                    </div>
    <div class="form-group">
        <label for="addAutoInput">Habitación</label>
        <input type="text" class="form-control" id="addAutoInput" placeholder="Buscar habitación..." required autocomplete="off">
        <div id="addAutoResults" class="search-results"></div>
        <input type="hidden" id="addAutoId" name="id_auto"> <!-- Campo oculto para el ID del auto -->
    </div>
    <div class="form-row">
    <div class="form-group col-md-6">
    <label for="addFechaRenta">Fecha Check-in</label>
    <input type="datetime-local" class="form-control" id="addFechaRenta" name="fecha_renta" required>
</div>
<div class="form-group col-md-6">
    <label for="addFechaDevolucion">Fecha Check-out</label>
    <input type="datetime-local" class="form-control" id="addFechaDevolucion" name="fecha_devolucion" required>
</div>


</div>
<div class="form-row">
    <div class="form-group col-md-6">
        <label for="addDias">Estadía</label>
        <input type="text" class="form-control" id="addDias" name="dias" readonly>
    </div>
    <div class="form-group col-md-6">
        <label for="addPrecioTotal">Precio Total</label>
        <input type="number" class="form-control" id="addPrecioTotal" name="precio_total" step="0.01" required autocomplete="off">
    </div>
</div>

                    <div class="form-group">
                        <label for="addEstado">Estado</label>
                        <select class="form-control" id="addEstado" name="estado" required autocomplete="off">
                            <option value="Reservado">Reservada</option>
                            <option value="Check-in">Check-in</option>
                            <option value="Check-out">Check-out</option>
                        </select>
                    </div>
                    <button type="submit" class="btn btn-primary">Agregar Reserva</button>
                </form>
            </div>
        </div>
    </div>
</div>

<!-- Modal de edición -->
<div class="modal fade" id="editModal" tabindex="-1" role="dialog" aria-labelledby="editModalLabel" aria-hidden="true">
    <div class="modal-dialog" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="editModalLabel">Editar Reservacción</h5>
                <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
            <div class="modal-body">
                <form id="editForm">
                <input type="hidden" name="id_renta" id="editId">
                <div class="form-group">
        <label for="editClienteInput">Cliente</label>
        <input type="text" class="form-control" id="editClienteInput" placeholder="Buscar cliente..." required autocomplete="off">
        <div id="editClienteResults" class="search-results"></div>
        <input type="hidden" id="editClienteId" name="id_cliente"> <!-- Campo oculto para el ID del cliente -->
    </div>
    <div class="form-group">
        <label for="editAutoInput">Auto</label>
        <input type="text" class="form-control" id="editAutoInput" placeholder="Buscar auto..." required autocomplete="off">
        <div id="editAutoResults" class="search-results"></div>
        <input type="hidden" id="editAutoId" name="id_auto"> <!-- Campo oculto para el ID del auto -->
    </div>
    <div class="form-row">
    <div class="form-group col-md-6">
        <label for="editFechaRenta">Fecha Check-in</label>
        <input type="datetime-local" class="form-control" id="editFechaRenta" name="fecha_renta" required>
    </div>
    <div class="form-group col-md-6">
        <label for="editFechaDevolucion">Fecha Check-out</label>
        <input type="datetime-local" class="form-control" id="editFechaDevolucion" name="fecha_devolucion" required>
    </div>
</div>

<div class="form-row">
    <div class="form-group col-md-6">
        <label for="editDias">Estadía</label>
        <input type="text" class="form-control" id="editDias" name="dias" readonly>
    </div>
    <div class="form-group col-md-6">
        <label for="editPrecioTotal">Precio Total</label>
        <input type="number" class="form-control" id="editPrecioTotal" name="precio_total" step="0.01" required autocomplete="off">
    </div>
</div>

                    <div class="form-group">
                        <label for="editEstado">Estado</label>
                        <select class="form-control" id="editEstado" name="estado" required autocomplete="off">
                            <option value="Reservado">Reservada</option>
                            <option value="Check-in">Check-in</option>
                            <option value="Check-out">Check-out</option>
                        </select>
                    </div>
                    <button type="submit" class="btn btn-primary">Guardar cambios</button>
                </form>
            </div>
        </div>
    </div>
</div>

<!-- Modal para detalles de la reservación -->
<div class="modal fade" id="detailsModal" tabindex="-1" role="dialog" aria-labelledby="detailsModalLabel" aria-hidden="true">
    <div class="modal-dialog" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="detailsModalLabel">Detalles de la Reservación</h5>
                <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
            <div class="modal-body">
                <div id="reservationDetails"></div>
                <form id="paymentForm">
                    <input type="hidden" id="id_cliente">
                    <input type="hidden" id="id_renta">
                    <div class="form-group">
                        <label for="amount">Monto a Pagar:</label>
                        <input type="number" step="0.01" class="form-control" id="amount" required>
                    </div>
                    <div class="form-group">
                        <label for="metodo_pago">Método de Pago:</label>
                        <select class="form-control" id="metodo_pago" required>
                            <option value="Efectivo">Efectivo</option>
                            <option value="Tarjeta de Crédito">Tarjeta de Crédito</option>
                            <option value="Transferencia Bancaria">Transferencia Bancaria</option>
                            <option value="Otro">Otro</option>
                        </select>
                    </div>
                    <button id="submitButton" type="submit" class="btn btn-primary">Realizar Pago</button>
                </form>
            </div>
        </div>
    </div>
</div>

<div class="modal fade" id="modalAddCliente" tabindex="-1" role="dialog" aria-labelledby="modalAddClienteLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg" role="document"> <!-- modal-lg para tamaño grande -->
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="modalAddClienteLabel">Agregar Nuevo Cliente</h5>
                <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
            <div class="modal-body">
                <form id="addClienteForm">
                    <div class="row">
                        <!-- Nombre -->
                        <div class="form-group col-md-6">
                            <label for="clienteNombre" class="form-label">Nombre</label>
                            <input type="text" class="form-control" id="clienteNombre" placeholder="Nombre" required autocomplete="off">
                        </div>
                        <!-- Apellido -->
                        <div class="form-group col-md-6">
                            <label for="clienteApellido" class="form-label">Apellido</label>
                            <input type="text" class="form-control" id="clienteApellido" placeholder="Apellido" required autocomplete="off">
                        </div>
                    </div>
                    
                    <div class="row">
                        <!-- Correo Electrónico -->
                        <div class="form-group col-md-6">
                            <label for="clienteEmail" class="form-label">Correo Electrónico</label>
                            <input type="email" class="form-control" id="clienteEmail" placeholder="Correo Electrónico" autocomplete="off">
                        </div>
                        <!-- Teléfono -->
                        <div class="form-group col-md-6">
                            <label for="clienteTelefono" class="form-label">Teléfono</label>
                            <input type="text" class="form-control" id="clienteTelefono" placeholder="Teléfono" autocomplete="off">
                        </div>
                    </div>
                    
                    <div class="row">
                        <!-- Documento Personal -->
                        <div class="form-group col-md-6">
                            <label for="clienteDocumento" class="form-label">Documento Personal</label>
                            <input type="text" class="form-control" id="clienteDocumento" placeholder="Documento (DNI, Pasaporte)" autocomplete="off">
                        </div>
                    </div>

                    <!-- Botón Guardar -->
                    <div class="text-end mt-3">
                        <button type="submit" class="btn btn-success">
                            <i class="fas fa-save"></i> Guardar
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>




    </div>

    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/@popperjs/core@2.11.6/dist/umd/popper.min.js"></script>
    <script src="https://maxcdn.bootstrapcdn.com/bootstrap/4.5.2/js/bootstrap.min.js"></script>
    <!-- SweetAlert2 JS -->
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    <script src="rentas.js"></script>
    <script> 

            $(document).ready(function() {
                $('#filterButton').click(function() {
                    var fechaDevolucion = $('#fechaDevolucionInput').val();
                    $.ajax({
                        url: 'filtrar_rentas.php',
                        type: 'POST',
                        data: { fecha_devolucion: fechaDevolucion },
                        success: function(response) {
                            $('#rentaTableBody').html(response);
                        }
                    });
                });

                $('#enviarRecordatorio').click(function() {
                var $boton = $(this); // Guardar referencia al botón
                var originalHtml = $boton.html(); // Guardar el HTML original del botón

                // Deshabilitar el botón y cambiar el texto a "Enviando..."
                $boton.prop('disabled', true);
                $boton.html('<i class="fas fa-spinner fa-spin"></i> Enviando...'); // Mostrar spinner animado y nuevo texto

                $.ajax({
                    url: 'enviar_recordatorio.php', // Archivo PHP que enviará los correos
                    method: 'POST',
                    success: function(response) {
                        // Mostrar alerta de éxito
                        Swal.fire({
                            icon: 'success',
                            title: '¡Éxito!',
                            text: 'Los recordatorios han sido enviados correctamente.',
                            confirmButtonText: 'Aceptar',
                            confirmButtonColor: '#3085d6',
                            background: '#f9f9f9', // Fondo personalizado
                        });

                        // Rehabilitar el botón y restaurar su contenido original
                        $boton.prop('disabled', false);
                        $boton.html(originalHtml); // Restaurar el icono y el texto original
                    },
                    error: function() {
                        // Mostrar alerta de error
                        Swal.fire({
                            icon: 'error',
                            title: 'Error',
                            text: 'Hubo un problema al enviar los correos.',
                            confirmButtonText: 'Aceptar',
                            confirmButtonColor: '#d33',
                            background: '#f9f9f9',
                        });

                        // Rehabilitar el botón y restaurar su contenido original incluso en caso de error
                        $boton.prop('disabled', false);
                        $boton.html(originalHtml); // Restaurar el icono y el texto original
                    }
                });
            });

            $('#addClienteButton').on('click', function () {
        // Abrir el modal para agregar cliente
        $('#modalAddCliente').modal('show');
    });

    $('#addClienteForm').on('submit', function (e) {
        e.preventDefault(); // Evitar envío del formulario

        // Serializar los datos del formulario
        let formData = {
            nombre: $('#clienteNombre').val(),
            apellido: $('#clienteApellido').val(),
            documento: $('#clienteDocumento').val(),
            telefono: $('#clienteTelefono').val(),
            email: $('#clienteEmail').val()
        };

        // Enviar datos mediante AJAX
        $.ajax({
            url: 'addCliente.php', // Ruta del archivo PHP
            type: 'POST',
            data: formData,
            dataType: 'json',
            success: function (response) {
                if (response.status === 'success') {
                    // Mostrar mensaje de éxito con Swal
                    Swal.fire({
                        icon: 'success',
                        title: '¡Cliente agregado!',
                        text: response.message,
                        showConfirmButton: false,
                        timer: 2000
                    });

                    // Cerrar el modal después del mensaje
                    $('#modalAddCliente').modal('hide');

                    // Limpiar el formulario
                    $('#addClienteForm')[0].reset();

                    // Actualizar automáticamente el cliente en los campos correspondientes
                    $('#addClienteInput').val(response.data.nombre + ' ' + response.data.apellido); // Mostrar nombre y apellido
                    $('#addClienteId').val(response.data.id_cliente); // Guardar el ID del cliente
                } else {
                    // Mostrar mensaje de error con Swal
                    Swal.fire({
                        icon: 'error',
                        title: 'Error',
                        text: response.message,
                        confirmButtonText: 'Entendido'
                    });
                }
            },
            error: function () {
                // Manejar errores de la solicitud AJAX
                Swal.fire({
                    icon: 'error',
                    title: 'Error en la solicitud',
                    text: 'No se pudo procesar la solicitud. Inténtalo nuevamente.',
                    confirmButtonText: 'Entendido'
                });
            }
        });
    });


            });
    
    </script>
    


</body>
</html>


</body>
</html>
