<?php
session_start();
require '../config.php';
require '../functions.php';

if (!isset($_SESSION['usuario_id'])) {
    header('Location: ../index.php');
    exit;
}

// Verifica si el usuario tiene permiso para acceder a la página
if (!usuarioTienePermiso($pdo, $_SESSION['usuario_id'], 'ver_caja')) {
    header('Location: ../sin_permiso.php');
    exit;
}
// Verifica los permisos del usuario para ver, editar y eliminar
$usuario_id = $_SESSION['usuario_id'];
$permisos = [
    'ver' => usuarioTienePermiso($pdo, $usuario_id, 'ver_caja'),
    'editar' => usuarioTienePermiso($pdo, $usuario_id, 'editar_caja'),
    'eliminar' => usuarioTienePermiso($pdo, $usuario_id, 'borrar_caja'),
];

function getCajas($pdo, $search = '', $offset = 0, $limit = 10) {
    $sql = "SELECT caja.*, usuarios.nombre, usuarios.apellido 
            FROM caja 
            JOIN usuarios ON caja.usuario_id = usuarios.id_usuario 
            WHERE caja.tipo LIKE :search
            ORDER BY caja.id_caja DESC
            LIMIT :offset, :limit";
    $stmt = $pdo->prepare($sql);
    $stmt->bindValue(':search', "%$search%", PDO::PARAM_STR);
    $stmt->bindValue(':offset', (int)$offset, PDO::PARAM_INT);
    $stmt->bindValue(':limit', (int)$limit, PDO::PARAM_INT);
    $stmt->execute();
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}


$search = isset($_POST['search']) ? $_POST['search'] : '';
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$limit = 10;
$offset = ($page - 1) * $limit;

// Obtener los registros de caja
$cajas = getCajas($pdo, $search, $offset, $limit);

// Contar el total de registros para la paginación
$totalSql = "SELECT COUNT(*) FROM caja WHERE tipo LIKE :search";
$totalStmt = $pdo->prepare($totalSql);
$totalStmt->execute(['search' => "%$search%"]);
$totalRows = $totalStmt->fetchColumn();
$totalPages = ceil($totalRows / $limit);
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>CRUD de Caja - Renta de Autos</title>
    <link href="https://maxcdn.bootstrapcdn.com/bootstrap/4.5.2/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css" rel="stylesheet">
    <style>
        .main-content {
            padding: 2rem;
        }

        .btn-primary {
            border-radius: 50px;
            font-size: 0.875rem;
        }

        .input-group {
            width: 100%;
            max-width: 500px;
        }

        .form-control {
            border-radius: 50px 0 0 50px;
            box-shadow: none;
        }

        .input-group-append .btn-secondary {
            border-radius: 0 50px 50px 0;
        }

        .btn-info {
            border-radius: 50px;
            font-size: 0.875rem;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
        }
        #message {
            position: fixed;
            top: 10px;
            right: 10px;
            width: 300px;
            z-index: 1050;
        }
        .modal-content {
            padding: 20px;
        }
        /* Estilos personalizados para el contenedor flotante */
        #floatingTicket {
            position: fixed;
            top: 20px;
            right: 20px;
            width: 360px;
            background-color: white;
            border: 1px solid #dee2e6;
            box-shadow: 0 4px 8px rgba(0, 0, 0, 0.2);
            z-index: 1000;
            display: none; /* Oculto por defecto */
            max-height: 90vh;
            overflow: auto;
            border-radius: 5px;
        }

        #floatingTicketHeader {
            cursor: move;
            padding: 0.75rem 1.25rem;
            background-color: #007bff;
            color: white;
        }

        #floatingTicketContent {
            padding: 10px;
        }

        .btn-show-ticket {
            position: fixed;
            bottom: 20px;
            right: 20px;
            z-index: 1000;
        }
    </style>
</head>
<body>
    <?php include '../menu.php'; ?>

    <div class="container mt-4">
       
        <div class="d-flex align-items-center justify-content-center p-3 my-3 text-white bg-dark rounded shadow-sm">
    <div class="lh-1 text-center">
        <h2 class="mb-4 text-center">
            <i class="fas fa-cash-register"></i> <!-- Icono de FontAwesome -->
            Administrar Caja
        </h2>
    </div>
</div>

<!-- Contenedor flex para el botón y los filtros, distribuidos en todo el ancho -->
<div class="d-flex flex-wrap align-items-center mb-3 w-100">
    <!-- Botón para agregar un nuevo ingreso/egreso -->
    <div class="form-group mb-2 mr-3" style="flex: 1;">
        <button class="btn btn-primary w-100" data-toggle="modal" data-target="#addModal">
            <i class="fas fa-plus"></i> Apertura/Cierre de Caja
        </button>
    </div>

    <!-- Filtros -->
    <div class="form-group mb-2 mr-3" style="flex: 1;">
        <label for="startDate" class="sr-only">Fecha de Inicio</label>
        <input type="date" class="form-control w-100" id="startDate">
    </div>

    <div class="form-group mb-2 mr-3" style="flex: 1;">
        <label for="endDate" class="sr-only">Fecha de Fin</label>
        <input type="date" class="form-control w-100" id="endDate">
    </div>

    <div class="form-group mb-2 mr-3 d-flex align-items-center" style="flex: 2;">
    <label for="type" class="mr-2 mb-0">Tipo:</label>
    <div class="input-group w-100">
        <select class="form-control" id="type">
            <option value="" disabled selected>Tipo</option>
            <option value="">Todos</option>
            <option value="apertura">Apertura</option>
            <option value="cierre">Cierre</option>
        </select>
        <div class="input-group-append">
            <button class="btn btn-secondary" type="button" id="filterBtn">
                <i class="fas fa-filter"></i> Filtrar
            </button>
        </div>
    </div>
</div>

    <!-- Botón para generar el PDF del informe completo -->
    <div class="form-group mb-2 mr-3" style="flex: 1;">
        <a id="generatePdf" class="btn btn-info w-100" target="_blank">
            <i class="fas fa-file-pdf"></i> Generar PDF
        </a>
    </div>

    <!-- Botón para mostrar el ticket flotante -->
    <div class="form-group mb-2" style="flex: 1;">
        <button class="btn btn-info w-100" onclick="toggleTicket()"> 
            <i class="fas fa-eye"></i> Ticket
        </button>
    </div>
</div>


    
<script>
function updatePdfLink() {
    var fechaInicio = encodeURIComponent(document.getElementById('startDate').value);
    var fechaFin = encodeURIComponent(document.getElementById('endDate').value);
    var tipo = encodeURIComponent(document.getElementById('type').value);  // Agregar filtro de tipo
    
    var pdfLink = '../reportes/caja_pdf.php?fecha_inicio=' + fechaInicio + '&fecha_fin=' + fechaFin + '&tipo=' + tipo;
    document.getElementById('generatePdf').setAttribute('href', pdfLink);
}

// Actualizar el enlace del PDF al cargar la página
updatePdfLink();

// Actualizar el enlace del PDF cuando cambien los filtros
document.getElementById('startDate').addEventListener('change', updatePdfLink);
document.getElementById('endDate').addEventListener('change', updatePdfLink);
document.getElementById('type').addEventListener('change', updatePdfLink);
</script>

            <div id="noResultsMessage" class="alert alert-warning d-none" role="alert">
                No se encontraron registros.
            </div>
            <table class="table table-bordered table-striped">
    <thead class="thead-dark">
        <tr>
            <th>ID</th>
            <th>Tipo</th>
            <th>Monto</th>
            <th>Fecha</th>
            <th>Usuario</th>
            <th>Acciones</th>
        </tr>
    </thead>
    <tbody id="cashTableBody">
        <?php foreach ($cajas as $caja): ?>
            <tr>
                <td><?php echo htmlspecialchars($caja['id_caja']); ?></td>
                <td>
                    <?php
                    // Definir el badge basado en el tipo de caja
                    $badgeClass = $caja['tipo'] == 'apertura' ? 'badge-success' : 'badge-danger';
                    $badgeText = $caja['tipo'] == 'apertura' ? 'Apertura' : 'Cierre';
                    ?>
                    <span class="badge <?php echo $badgeClass; ?>"><?php echo $badgeText; ?></span>
                </td>
                <td><?php echo htmlspecialchars(formatMonto($moneda, $caja['monto'])); ?></td>
                <td><?php echo htmlspecialchars($caja['fecha']); ?></td>
                <td><?php echo htmlspecialchars($caja['nombre']) . ' ' . htmlspecialchars($caja['apellido']); ?></td>
                <td>
                <?php if ($permisos['editar']) : ?>
                    <button class="btn btn-warning btn-sm edit-btn" data-id="<?php echo $caja['id_caja']; ?>" data-toggle="modal" data-target="#editModal"><i class="fas fa-edit"></i></button>
                    <?php endif; ?>
                    <?php if ($permisos['eliminar']) : ?>
                    <button class="btn btn-danger btn-sm delete-btn" data-id="<?php echo $caja['id_caja']; ?>"><i class="fas fa-trash-alt"></i></button>
                    <?php endif; ?>
                    <?php if ($caja['tipo'] == 'cierre'): ?>
                        <?php if ($permisos['ver']) : ?>
                    <button class="btn btn-info btn-sm reprint-btn" data-id="<?php echo $caja['id_caja']; ?>">
                        <i class="fas fa-print"></i>
                    </button>
                    <?php endif; ?>
                <?php endif; ?>

                </td>
            </tr>

            <!-- Contenedor flotante del ticket -->
            <div id="floatingTicket" class="card">
                <div id="floatingTicketHeader" class="card-header">
                    <span>Ticket de Caja</span>
                    <button id="closeButton" class="close text-white ml-2" onclick="toggleTicket()">&times;</button>
                </div>
                <div id="floatingTicketContent" class="card-body">
                <iframe src="print_ticket_flotante.php?id_caja=<?php echo $caja['id_caja']; ?>" frameborder="0" style="width: 100%; height: 100%;"></iframe>
                </div>
            </div>
        <?php endforeach; ?>
    </tbody>
</table>


            <!-- Paginación -->
            <nav aria-label="Page navigation">
                <ul class="pagination">
                    <li class="page-item <?php if ($page <= 1) echo 'disabled'; ?>">
                        <a class="page-link" href="?page=<?php echo max($page - 1, 1); ?>" aria-label="Anterior">
                            <span aria-hidden="true">&laquo;</span>
                        </a>
                    </li>
                    <?php for ($i = 1; $i <= $totalPages; $i++): ?>
                        <li class="page-item <?php if ($i == $page) echo 'active'; ?>">
                            <a class="page-link" href="?page=<?php echo $i; ?>"><?php echo $i; ?></a>
                        </li>
                    <?php endfor; ?>
                    <li class="page-item <?php if ($page >= $totalPages) echo 'disabled'; ?>">
                        <a class="page-link" href="?page=<?php echo min($page + 1, $totalPages); ?>" aria-label="Siguiente">
                            <span aria-hidden="true">&raquo;</span>
                        </a>
                    </li>
                </ul>
            </nav>
        

        <div id="message" class="alert d-none"></div>

        <!-- Modal para agregar caja -->
        <div class="modal fade" id="addModal" tabindex="-1" role="dialog" aria-labelledby="addModalLabel" aria-hidden="true">
            <div class="modal-dialog" role="document">
                <div class="modal-content">
                    <div class="modal-header">
                        <h5 class="modal-title" id="addModalLabel">Agregar Apertura/Cierre de Caja</h5>
                        <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                            <span aria-hidden="true">&times;</span>
                        </button>
                    </div>
                    <div class="modal-body">
                        <form id="addForm">
                            <div class="form-group">
                                <label for="addTipo">Tipo</label>
                                <select id="addTipo" name="tipo" class="form-control">
                                    <option value="apertura">Apertura</option>
                                    <option value="cierre">Cierre</option>
                                </select>
                            </div>
                            <div class="form-group">
                                <label for="addMonto">Monto</label>
                                <input type="number" id="addMonto" name="monto" class="form-control" step="0.01" required autocomplete="off">
                            </div>
                            <button type="submit" class="btn btn-primary">Guardar</button>
                        </form>
                    </div>
                </div>
            </div>
        </div>

        <!-- Modal para editar caja -->
        <div class="modal fade" id="editModal" tabindex="-1" role="dialog" aria-labelledby="editModalLabel" aria-hidden="true">
            <div class="modal-dialog" role="document">
                <div class="modal-content">
                    <div class="modal-header">
                        <h5 class="modal-title" id="editModalLabel">Editar Apertura/Cierre de Caja</h5>
                        <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                            <span aria-hidden="true">&times;</span>
                        </button>
                    </div>
                    <div class="modal-body">
                        <form id="editForm">
                            <input type="hidden" id="editId" name="id_caja">
                            <div class="form-group">
                                <label for="editTipo">Tipo</label>
                                <select id="editTipo" name="tipo" class="form-control">
                                    <option value="apertura">Apertura</option>
                                    <option value="cierre">Cierre</option>
                                </select>
                            </div>
                            <div class="form-group">
                                <label for="editMonto">Monto</label>
                                <input type="number" id="editMonto" name="monto" class="form-control" step="0.01" required autocomplete="off">
                            </div>
                            <button type="submit" class="btn btn-primary">Guardar Cambios</button>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/@popperjs/core@2.11.6/dist/umd/popper.min.js"></script>
    <script src="https://maxcdn.bootstrapcdn.com/bootstrap/4.5.2/js/bootstrap.min.js"></script>
    <script src="caja.js"></script>
    <script>
   function toggleTicket() {
        var ticket = document.getElementById('floatingTicket');
        // Revisar el valor de 'display' en el estilo del ticket
        if (ticket.style.display === 'none' || ticket.style.display === '') {
            ticket.style.display = 'block';  // Mostrar el ticket si está oculto
        } else {
            ticket.style.display = 'none';  // Ocultar el ticket si está visible
        }
    }

    // Funcionalidad para arrastrar el ticket flotante
    (function() {
        var header = document.getElementById("floatingTicketHeader");
        var ticket = document.getElementById("floatingTicket");
        var offsetX = 0, offsetY = 0, posX = 0, posY = 0;

        header.onmousedown = function(e) {
            e.preventDefault();
            posX = e.clientX;
            posY = e.clientY;
            document.onmouseup = closeDragElement;
            document.onmousemove = elementDrag;
        };

        function elementDrag(e) {
            e.preventDefault();
            offsetX = posX - e.clientX;
            offsetY = posY - e.clientY;
            posX = e.clientX;
            posY = e.clientY;
            ticket.style.top = (ticket.offsetTop - offsetY) + "px";
            ticket.style.left = (ticket.offsetLeft - offsetX) + "px";
        }

        function closeDragElement() {
            document.onmouseup = null;
            document.onmousemove = null;
        }
    })();
</script>

</body>
</html>
