<?php
session_start();
require '../config.php';
require '../functions.php';

// Verificar si la caja está aperturada
if (!isCajaAperturada($pdo)) {
    header('Location: ../caja_no_aperturada.php');
    exit;
}

// Verifica si el usuario está autenticado
if (!isset($_SESSION['usuario_id'])) {
    header('Location: ../index.php');
    exit;
}

date_default_timezone_set('America/El_Salvador');
$fecha_actual = date('Y-m-d');
$hora_actual = date('H:i:s');

// Obtener el turno activo
$sql_turno = "SELECT id_turno, nombre FROM turnos WHERE hora_inicio <= ? AND hora_fin >= ? LIMIT 1";
$stmt_turno = $pdo->prepare($sql_turno);
$stmt_turno->execute([$hora_actual, $hora_actual]);
$turno = $stmt_turno->fetch(PDO::FETCH_ASSOC);
$id_turno = $turno['id_turno'] ?? null;
$turno_actual = $turno ? $turno['nombre'] : 'No hay turno activo';

$ventas = [];
$total_rifa_80 = 0;
$total_rifa_70 = 0;
$numero_mayor_venta = 0;
$mayor_venta = 0;

if ($id_turno !== null) {
    $sql_ventas = "SELECT 
                        numero, 
                        SUM(CASE WHEN rifa = 80 THEN ganancia ELSE 0 END) AS ventas_rifa_80,
                        SUM(CASE WHEN rifa = 70 THEN ganancia ELSE 0 END) AS ventas_rifa_70
                   FROM ventas 
                   WHERE id_turno = ? AND DATE(fecha) = ? 
                   GROUP BY numero";

    $stmt_ventas = $pdo->prepare($sql_ventas);
    $stmt_ventas->execute([$id_turno, $fecha_actual]);

    while ($row = $stmt_ventas->fetch(PDO::FETCH_ASSOC)) {
        $ventas[$row['numero']] = [
            'rifa_80' => $row['ventas_rifa_80'],
            'rifa_70' => $row['ventas_rifa_70'],
            'total' => $row['ventas_rifa_80'] + $row['ventas_rifa_70']
        ];
        $total_rifa_80 += $row['ventas_rifa_80'];
        $total_rifa_70 += $row['ventas_rifa_70'];

        if ($ventas[$row['numero']]['total'] > $mayor_venta) {
            $mayor_venta = $ventas[$row['numero']]['total'];
            $numero_mayor_venta = $row['numero'];
        }
    }
}

$total_general = $total_rifa_80 + $total_rifa_70;

// Ordenar los números de mayor a menor en base al total_numero
uasort($ventas, function ($a, $b) {
    return $b['total'] - $a['total'];
});

// Informacion de Ventas 
$venta_general = 0;
$comision_total = 0;
$liquido_total = 0;

if ($id_turno !== null) {
    $sql_ventas = "SELECT SUM(valor) AS venta_general, SUM(comision) AS comision_total 
                   FROM ventas 
                   WHERE id_turno = ? AND DATE(fecha) = ?";

    $stmt_ventas = $pdo->prepare($sql_ventas);
    $stmt_ventas->execute([$id_turno, $fecha_actual]);
    $row = $stmt_ventas->fetch(PDO::FETCH_ASSOC);

    $venta_general = $row['venta_general'] ?? 0;
    $comision_total = $row['comision_total'] ?? 0;
    $liquido_total = $venta_general - $comision_total;
}
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Principal</title>
    <link rel="shortcut icon" href="../img/favi.ico">
    <link rel="stylesheet" href="https://maxcdn.bootstrapcdn.com/bootstrap/4.5.2/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <style>
        table {
            width: 100%;
            border-collapse: collapse;
        }
        th, td {
            border: 1px solid black;
            text-align: center;
            padding: 10px;
        }
        .badge-number {
            font-size: 1.5rem;
            font-weight: bold;
            display: block;
            margin-bottom: 10px;
            color: #333;
        }
        .text-rifa-80, .text-rifa-70 {
            font-size: 1rem;
            font-weight: bold;
        }
        .text-rifa-80 {
            color: blue;
        }
        .text-rifa-70 {
            color: red;
        }
        .total-numero {
            color: green;
            font-weight: bold;
            font-size: 1.1rem;
        }
        .total-box {
            background-color: #f8f9fa;
            font-weight: bold;
            padding: 10px;
            border-radius: 8px;
            text-align: center;
        }
        .numero-mayor {
            background-color: yellow;
            font-weight: bold;
            font-size: 1.5rem;
        }
        .text-center {
            text-align: center;
        }
        .stat-box {
            background-color: #f8f9fa;
            padding: 20px;
            border-radius: 10px;
            box-shadow: 0 4px 8px rgba(0, 0, 0, 0.1);
            text-align: center;
            margin-bottom: 20px;
            font-weight: bold;
        }
        .stat-box h3 {
            font-size: 2rem;
            color: #ffffff;
        }
        .stat-box .value {
            font-size: 1.5rem;
            font-weight: bold;
            color: #333;
        }
        .stat-box .description {
            font-size: 1rem;
            color: #666;
        }
        .stat-box.venta {
            background-color: #00a4bd;
        }
        .stat-box.comision {
            background-color: #ed0034;
        }
        .stat-box.liquido {
            background-color: #28a745;
        }
    </style>
</head>
 <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/@popperjs/core@2.11.6/dist/umd/popper.min.js"></script>
    <script src="https://maxcdn.bootstrapcdn.com/bootstrap/4.5.2/js/bootstrap.min.js"></script>
<body>
    <?php include '../menu.php'; ?>

    <div class="container mt-4">
        <h4 class="text-center">Ventas del Día: <?php echo fecha_en_espanol(); ?></h4>

        <?php if ($id_turno === null): ?>
            <div class="alert alert-danger text-center" role="alert">
                ❌ No hay turnos activos en este momento.
            </div>
        <?php else: ?>

            <div class="row">
                <div class="col-md-4">
                    <div class="stat-box venta">
                        <h3>Venta General</h3>
                        <div class="value">$<?php echo number_format($venta_general, 2, '.', ','); ?></div>
                        
                    </div>
                </div>
                <div class="col-md-4">
                    <div class="stat-box comision">
                        <h3>Comisión</h3>
                        <div class="value">$<?php echo number_format($comision_total, 2, '.', ','); ?></div>
                        
                    </div>
                </div>
                <div class="col-md-4">
                    <div class="stat-box liquido">
                        <h3>Líquido</h3>
                        <div class="value">$<?php echo number_format($liquido_total, 2, '.', ','); ?></div>
                        
                    </div>
                </div>
            </div>
            
            <div class="table-responsive">
                <table class="table table-bordered text-center">
                    <thead class="table-dark">
                        <tr>
                            <th colspan="10"><h5>Sorteo Actual: <?php echo $turno_actual; ?>  </h5></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php
                        $contador = 0;
                        foreach ($ventas as $numero => $datos) {
                            if ($contador % 4 == 0) {
                                if ($contador != 0) {
                                    echo "</tr>";
                                }
                                echo "<tr>";
                            }

                            $ventas_rifa_80 = $datos['rifa_80'];
                            $ventas_rifa_70 = $datos['rifa_70'];
                            $total_numero = $datos['total'];

                           //$clase_resaltado = ($numero == $numero_mayor_venta) ? 'numero-mayor' : '';
                           $clase_resaltado = '';

                            echo "<td class='$clase_resaltado'>
                                    <span class='badge-number'>
                                        " . sprintf("%02d", $numero) . "
                                    </span>
                                    <span class='text-rifa-80'>$" . number_format($ventas_rifa_80, 2, '.', ',') . "</span> <span class='text-rifa-70'>$" . number_format($ventas_rifa_70, 2, '.', ',') . "</span><br>
                                    <span class='total-numero'>$" . number_format($total_numero, 2, '.', ',') . "</span>
                                </td>";

                            $contador++;
                        }
                        // Cerrar la última fila si es necesario
                        if ($contador % 10 != 0) {
                            echo "</tr>";
                        }
                        ?>
                    </tbody>
                </table>
           
            </div>
            <div class="row mt-4">
                <div class="col-md-4 offset-md-2">
                    <div class="total-box">
                        🔵 <strong>Venta 80:</strong> $<?php echo number_format($total_rifa_80, 2, '.', ','); ?>
                    </div>
                </div>
                <div class="col-md-4">
                    <div class="total-box">
                        🔴 <strong>Venta 70:</strong> $<?php echo number_format($total_rifa_70, 2, '.', ','); ?>
                    </div>
                </div>
               <div class="col-md-8 offset-md-2 mt-3">
                    <div class="total-box" style="background-color: #e9ecef;">
                        <strong>Total Venta:</strong> $<?php echo number_format($total_general, 2, '.', ','); ?>
                    </div>
                </div>
            </div>
        <?php endif; ?>
    </div>
</body>
</html>