<?php
require '../config.php';
require '../functions.php';
session_start();

// Verificar si la sesión está activa y obtener el valor de $fifa
if (!isset($_SESSION['rifa'])) {
    echo json_encode(['status' => 'error', 'message' => 'La sesión no está activa o falta el valor de la rifa.']);
    exit;
}
$fifa = $_SESSION['rifa'];
$comisionUser = $_SESSION['comision']; // Corregir el nombre de la variable
$comisionPorcentaje = $comisionUser / 100; // Convertir a decimal

// Obtener el ID del vendedor y el ID del turno
$id_vendedor = $_SESSION['usuario_id'] ?? null;
$id_turno = $_POST['id_turno'] ?? null;

// Verificar si se recibieron los datos requeridos
if (!$id_vendedor || !$id_turno) {
    echo json_encode(['status' => 'error', 'message' => 'Faltan datos requeridos: ID del vendedor o ID del turno.']);
    exit;
}

// Obtener la fecha y hora actual
$fecha_actual = date('Y-m-d');
$hora_actual = date('H:i:s');

try {
    // Verificar si el turno está activo
    $sql_turno = "SELECT hora_inicio, hora_fin FROM turnos WHERE id_turno = :id_turno";
    $stmt_turno = $pdo->prepare($sql_turno);
    $stmt_turno->execute([':id_turno' => $id_turno]);
    $turno = $stmt_turno->fetch(PDO::FETCH_ASSOC);

    if (!$turno || $hora_actual < $turno['hora_inicio'] || $hora_actual > $turno['hora_fin']) {
        echo json_encode(['status' => 'error', 'message' => 'El turno no está activo en este momento.']);
        exit;
    }

    // Obtener los productos del carrito para este vendedor y turno
    $sql_carrito = "SELECT numero, cliente, valor FROM carrito WHERE id_vendedor = :id_vendedor AND id_turno = :id_turno";
    $stmt_carrito = $pdo->prepare($sql_carrito);
    $stmt_carrito->execute([
        ':id_vendedor' => $id_vendedor,
        ':id_turno' => $id_turno
    ]);
    $carrito = $stmt_carrito->fetchAll(PDO::FETCH_ASSOC);

    if (empty($carrito)) {
        echo json_encode(['status' => 'error', 'message' => 'No hay productos en el carrito para vender.']);
        exit;
    }

    // Iniciar transacción para garantizar consistencia de datos
    $pdo->beginTransaction();

    // Obtener el último correlativo utilizado
    $sql_correlativo = "SELECT MAX(correlativo) as ultimo_correlativo FROM ventas WHERE id_vendedor = :id_vendedor AND id_turno = :id_turno";
    $stmt_correlativo = $pdo->prepare($sql_correlativo);
    $stmt_correlativo->execute([
        ':id_vendedor' => $id_vendedor,
        ':id_turno' => $id_turno
    ]);
    $correlativo_data = $stmt_correlativo->fetch(PDO::FETCH_ASSOC);
    $correlativo = ($correlativo_data['ultimo_correlativo'] ?? 0) + 1;

    // Insertar cada producto del carrito en la tabla ventas
    $sql_insert = "INSERT INTO ventas (id_vendedor, id_turno, numero, cliente, valor, ganancia, fecha, correlativo, rifa, comision) 
                   VALUES (:id_vendedor, :id_turno, :numero, :cliente, :valor, :ganancia, NOW(), :correlativo, :rifa, :comision)";
    $stmt_insert = $pdo->prepare($sql_insert);

    foreach ($carrito as $item) {
        // Validar y convertir el valor a flotante
        if (!is_numeric($item['valor'])) {
            throw new Exception("El valor del producto no es numérico: " . $item['valor']);
        }
        $valor = floatval($item['valor']);

        // Calcular la ganancia según el valor de $fifa
        if ($fifa == 70) {
            $ganancia = calcularGanancia($valor);
        } elseif ($fifa == 80) {
            $ganancia = calcularGanancia2($valor);
        } else {
            $ganancia = 0; // Valor por defecto si $fifa no es 70 ni 80
        }

        // Calcular la comisión
        $comision = $valor * $comisionPorcentaje;

        // Insertar los datos en la tabla ventas
        $stmt_insert->execute([
            ':id_vendedor' => $id_vendedor,
            ':id_turno' => $id_turno,
            ':numero' => $item['numero'],
            ':cliente' => $item['cliente'],
            ':valor' => $valor,
            ':ganancia' => $ganancia,
            ':correlativo' => $correlativo,
            ':rifa' => $fifa,
            ':comision' => $comision
        ]);
    }

    // Vaciar el carrito después de guardar las ventas
    $sql_delete = "DELETE FROM carrito WHERE id_vendedor = :id_vendedor AND id_turno = :id_turno";
    $stmt_delete = $pdo->prepare($sql_delete);
    $stmt_delete->execute([
        ':id_vendedor' => $id_vendedor,
        ':id_turno' => $id_turno
    ]);

    // Confirmar la transacción
    $pdo->commit();

    // Obtener los datos de la venta recién creada y combinar con usuarios y turnos
    $sql_venta = "
        SELECT v.*, u.nombre AS nombre_vendedor, v.correlativo, t.nombre AS nombre_turno
        FROM ventas v
        JOIN usuarios u ON v.id_vendedor = u.id_usuario
        JOIN turnos t ON v.id_turno = t.id_turno
        WHERE v.id_vendedor = :id_vendedor
        AND v.id_turno = :id_turno
        ORDER BY v.id_venta DESC
        LIMIT 1
    ";
    $stmt_venta = $pdo->prepare($sql_venta);
    $stmt_venta->execute([
        ':id_vendedor' => $id_vendedor,
        ':id_turno' => $id_turno
    ]);
    $venta = $stmt_venta->fetch(PDO::FETCH_ASSOC);

    // Devolver respuesta con los datos de la venta, el nombre del vendedor y el nombre del turno
    echo json_encode([
        'status' => 'success',
        'message' => 'Venta realizada con éxito. Todos los productos del carrito han sido guardados en ventas.',
        'venta' => $venta
    ]);

} catch (PDOException $e) {
    // Si ocurre un error, se revierte la transacción
    $pdo->rollBack();
    echo json_encode(['status' => 'error', 'message' => 'Error en la venta: ' . $e->getMessage()]);
} catch (Exception $e) {
    // Capturar excepciones generales
    $pdo->rollBack();
    echo json_encode(['status' => 'error', 'message' => 'Error: ' . $e->getMessage()]);
}
?>