<?php
session_start();
require '../config.php';
require '../functions.php';

// Verificar si la caja está aperturada
if (!isCajaAperturada($pdo)) {
    header('Location: ../caja_no_aperturada.php');
    exit;
}

if (!isset($_SESSION['usuario_id'])) {
    header('Location: ../index.php');
    exit;
}

// Verifica si el usuario tiene permiso para acceder a la página
if (!usuarioTienePermiso($pdo, $_SESSION['usuario_id'], 'ver_efectivo')) {
    header('Location: ../sin_permiso.php');
    exit;
}

// Verifica los permisos del usuario para ver, editar y eliminar
$usuario_id = $_SESSION['usuario_id'];
$permisos = [
    'ver' => usuarioTienePermiso($pdo, $usuario_id, 'ver_efectivo'),
    'editar' => usuarioTienePermiso($pdo, $usuario_id, 'editar_efectivo'),
    'eliminar' => usuarioTienePermiso($pdo, $usuario_id, 'borrar_efectivo'),
];

function getEfectivos($pdo, $search = '', $offset = 0, $limit = 10) {
    $sql = "SELECT efectivo.*, usuarios.nombre_usuario, usuarios.apellido 
            FROM efectivo 
            JOIN usuarios ON efectivo.usuario_id = usuarios.id_usuario 
            WHERE efectivo.tipo LIKE :search
            ORDER BY efectivo.id_efectivo DESC
            LIMIT :offset, :limit";
    $stmt = $pdo->prepare($sql);
    $stmt->bindValue(':search', "%$search%", PDO::PARAM_STR);
    $stmt->bindValue(':offset', (int)$offset, PDO::PARAM_INT);
    $stmt->bindValue(':limit', (int)$limit, PDO::PARAM_INT);
    $stmt->execute();
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

$search = isset($_POST['search']) ? $_POST['search'] : '';
$page = isset($_POST['page']) ? (int)$_POST['page'] : 1;
$limit = 10;
$offset = ($page - 1) * $limit;

// Obtener los registros de efectivo
$efectivos = getEfectivos($pdo, $search, $offset, $limit);

// Contar el total de registros para la paginación
$totalSql = "SELECT COUNT(*) FROM efectivo WHERE tipo LIKE :search";
$totalStmt = $pdo->prepare($totalSql);
$totalStmt->execute(['search' => "%$search%"]);
$totalRows = $totalStmt->fetchColumn();
$totalPages = ceil($totalRows / $limit);
?>


<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>CRUD de Efectivo - Gestión de Ingresos y Egresos</title>
    <link href="https://maxcdn.bootstrapcdn.com/bootstrap/4.5.2/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css" rel="stylesheet">
    <style>
        .main-content {
            padding: 2rem;
        }

       
        .btn-primary {
            border-radius: 50px;
            font-size: 0.875rem;
        }

        .input-group {
            width: 100%;
            max-width: 500px;
        }

        .form-control {
            border-radius: 50px 0 0 50px;
            box-shadow: none;
        }

        .input-group-append .btn-secondary {
            border-radius: 0 50px 50px 0;
        }

        .btn-info {
            border-radius: 50px;
            font-size: 0.875rem;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
        }
        #message {
            position: fixed;
            top: 10px;
            right: 10px;
            width: 300px;
            z-index: 1050;
        }
        .modal-content {
            padding: 20px;
        }
        .filter-row {
        display: flex;
        align-items: flex-end;
        gap: 15px; /* Espacio entre los elementos */
    }
    .filter-row .form-group {
        flex: 1; /* Los campos de fecha ocupan el espacio disponible */
    }
    </style>
</head>
<body>
    <?php include '../menu.php'; ?>

    <div class="container mt-4">
        
        <div class="d-flex align-items-center justify-content-center p-3 my-3 text-white bg-dark rounded shadow-sm">
    <div class="lh-1 text-center">
        <h2 class="mb-4 text-center">
            <i class="fas fa-money-bill-wave"></i> <!-- Icono de FontAwesome -->
            Administrar Ingresos y Egresos de Efectivo
        </h2>
    </div>
</div>


            

            <div id="noResultsMessage" class="alert alert-warning d-none" role="alert">
                No se encontraron registros.
            </div>
            
<!-- Contenedor flex para el botón y los filtros -->
<div class="d-flex flex-wrap mb-3 align-items-center w-100">
    <!-- Botón para agregar un nuevo ingreso/egreso -->
    <div class="form-group mb-2 mr-4" style="flex: 1;">
        <button class="btn btn-primary w-100" data-toggle="modal" data-target="#addModal">
            <i class="fas fa-plus"></i> Ingreso/Egreso
        </button>
    </div>

    <!-- Filtros de fecha -->
    <div class="form-group mb-2 mr-4 d-flex align-items-center" style="flex: 1;">
        <label for="startDate" class="mr-2 mb-0">Fecha de Inicio:</label>
        <input type="date" class="form-control w-100" id="startDate">
    </div>
    <div class="form-group mb-2 mr-4 d-flex align-items-center" style="flex: 1;">
        <label for="endDate" class="mr-2 mb-0">Fecha de Fin:</label>
        <input type="date" class="form-control w-100" id="endDate">
    </div>

    <!-- Filtro de tipo -->
    <div class="form-group mb-2 mr-4 d-flex align-items-center" style="flex: 1;">
    <div class="input-group w-100">
    <label for="type" class="mr-2 mb-0">Tipo:</label>
        <select class="form-control" id="type">
        <option value="" disabled selected>Seleccionar</option>
            <option value="">Todos</option>
            <option value="ingreso">Ingreso</option>
            <option value="egreso">Egreso</option>
        </select>
        <div class="input-group-append">
            <button class="btn btn-secondary" type="button" id="filterBtn">
                <i class="fas fa-filter"></i> Filtrar
            </button>
        </div>
    </div>
    </div>

    <!-- Botón para generar el PDF del informe completo -->
    <div class="form-group mb-2" style="flex: 1;">
        <a id="generatePdf" class="btn btn-info w-100" target="_blank">
            <i class="fas fa-file-pdf"></i> Generar PDF
        </a>
    </div>
</div>



    
<script>
function updatePdfLink() {
    var fechaInicio = encodeURIComponent(document.getElementById('startDate').value);
    var fechaFin = encodeURIComponent(document.getElementById('endDate').value);
    var tipo = encodeURIComponent(document.getElementById('type').value);  // Agregar filtro de tipo
    
    var pdfLink = '../reportes/efectivo_pdf.php?fecha_inicio=' + fechaInicio + '&fecha_fin=' + fechaFin + '&tipo=' + tipo;
    document.getElementById('generatePdf').setAttribute('href', pdfLink);
}

// Actualizar el enlace del PDF al cargar la página
updatePdfLink();

// Actualizar el enlace del PDF cuando cambien los filtros
document.getElementById('startDate').addEventListener('change', updatePdfLink);
document.getElementById('endDate').addEventListener('change', updatePdfLink);
document.getElementById('type').addEventListener('change', updatePdfLink);
</script>




            <table class="table table-bordered table-striped">
    <thead class="thead-dark">
        <tr>
            <th>ID</th>
            <th>Tipo</th>
            <th>Monto</th>
            <th>Fecha</th>
            <th>Usuario</th>
            <th>Acciones</th>
        </tr>
    </thead>
    <tbody id="efectivoTableBody">
        <?php foreach ($efectivos as $efectivo): ?>
            <tr>
                <td><?php echo htmlspecialchars($efectivo['id_efectivo']); ?></td>
                <td>
                    <?php
                    $badgeClass = $efectivo['tipo'] == 'ingreso' ? 'badge-success' : 'badge-danger';
                    $badgeText = $efectivo['tipo'] == 'ingreso' ? 'Ingreso' : 'Egreso';
                    ?>
                    <span class="badge <?php echo $badgeClass; ?>"><?php echo $badgeText; ?></span>
                </td>
                <td><?php echo htmlspecialchars(formatMonto($moneda, $efectivo['monto'], 2)); ?></td>
                <td><?php echo htmlspecialchars($efectivo['fecha']); ?></td>
                <td><?php echo htmlspecialchars($efectivo['nombre_usuario']); ?></td>
                <td>
                <?php if ($permisos['editar']) : ?>
                    <button class="btn btn-warning btn-sm edit-btn" data-id="<?php echo $efectivo['id_efectivo']; ?>" data-toggle="modal" data-target="#editModal"><i class="fas fa-edit"></i></button>
                    <?php endif; ?>
                    <?php if ($permisos['eliminar']) : ?>
                    <button class="btn btn-danger btn-sm delete-btn" data-id="<?php echo $efectivo['id_efectivo']; ?>"><i class="fas fa-trash-alt"></i></button>
                    <?php endif; ?>
                </td>
            </tr>
        <?php endforeach; ?>
    </tbody>
</table>

<!-- Paginación -->
<nav aria-label="Page navigation">
  <ul class="pagination">
    <?php for ($i = 1; $i <= $totalPages; $i++): ?>
        <li class="page-item <?php echo $i == $page ? 'active' : ''; ?>">
            <a class="page-link" href="#" data-page="<?php echo $i; ?>"><?php echo $i; ?></a>
        </li>
    <?php endfor; ?>
  </ul>
</nav>

        <div id="message" class="alert d-none"></div>

        <!-- Modal para agregar efectivo -->
<div class="modal fade" id="addModal" tabindex="-1" role="dialog" aria-labelledby="addModalLabel" aria-hidden="true">
    <div class="modal-dialog" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="addModalLabel">Agregar Ingreso/Egreso de Efectivo</h5>
                <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
            <div class="modal-body">
                <form id="addForm">
                    <div class="form-group">
                        <label for="addTipo">Tipo</label>
                        <select id="addTipo" name="tipo" class="form-control">
                            <option value="ingreso">Ingreso</option>
                            <option value="egreso">Egreso</option>
                        </select>
                    </div>
                    <div class="form-group">
                        <label for="addMonto">Monto</label>
                        <input type="number" id="addMonto" name="monto" class="form-control" step="0.01" required autocomplete="off">
                    </div>
                    <div class="form-group">
                        <label for="addMotivo">Motivo</label>
                        <input type="text" id="addMotivo" name="motivo" class="form-control" required autocomplete="off">
                    </div>
                    <button type="submit" class="btn btn-primary">Guardar</button>
                </form>
            </div>
        </div>
    </div>
</div>


        <!-- Modal para editar efectivo -->
<div class="modal fade" id="editModal" tabindex="-1" role="dialog" aria-labelledby="editModalLabel" aria-hidden="true">
    <div class="modal-dialog" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="editModalLabel">Editar Ingreso/Egreso de Efectivo</h5>
                <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
            <div class="modal-body">
                <form id="editForm">
                    <input type="hidden" id="editId" name="id_efectivo">
                    <div class="form-group">
                        <label for="editTipo">Tipo</label>
                        <select id="editTipo" name="tipo" class="form-control">
                            <option value="ingreso">Ingreso</option>
                            <option value="egreso">Egreso</option>
                        </select>
                    </div>
                    <div class="form-group">
                        <label for="editMonto">Monto</label>
                        <input type="number" id="editMonto" name="monto" class="form-control" step="0.01" required autocomplete="off">
                    </div>
                    <div class="form-group">
                        <label for="editMotivo">Motivo</label>
                        <input type="text" id="editMotivo" name="motivo" class="form-control" required autocomplete="off">
                    </div>
                    <button type="submit" class="btn btn-primary">Guardar cambios</button>
                </form>
            </div>
        </div>
    </div>
</div>


    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/popper.js/1.16.0/umd/popper.min.js"></script>
    <script src="https://stackpath.bootstrapcdn.com/bootstrap/4.5.2/js/bootstrap.min.js"></script>
    <script src="efectivo.js"></script>
</body>
</html>
