<?php
session_start();
require '../config.php';
require '../functions.php';

// Verificar si la caja está aperturada
if (!isCajaAperturada($pdo)) {
    header('Location: ../caja_no_aperturada.php');
    exit;
}

// Verifica si el usuario está autenticado
if (!isset($_SESSION['usuario_id'])) {
    header('Location: ../index.php');
    exit;
}
// Verifica si el usuario tiene permiso para acceder a la página
if (!usuarioTienePermiso($pdo, $_SESSION['usuario_id'], 'ver_pagos')) {
    header('Location: ../sin_permiso.php');
    exit;
}
// Verifica los permisos del usuario para ver, editar y eliminar
$usuario_id = $_SESSION['usuario_id'];
$permisos = [
    'ver' => usuarioTienePermiso($pdo, $usuario_id, 'ver_pagos'),
    'editar' => usuarioTienePermiso($pdo, $usuario_id, 'editar_pagos'),
    'eliminar' => usuarioTienePermiso($pdo, $usuario_id, 'borrar_pagos'),
];
// Número de registros por página
$registrosPorPagina = 10;

// Obtener el número de página actual
$paginaActual = isset($_GET['pagina']) ? (int)$_GET['pagina'] : 1;
$paginaActual = $paginaActual > 0 ? $paginaActual : 1;

// Calcular el desplazamiento
$inicio = ($paginaActual - 1) * $registrosPorPagina;

// Obtener el rango de fechas
$fechaInicio = isset($_POST['fecha_inicio']) ? $_POST['fecha_inicio'] : '';
$fechaFin = isset($_POST['fecha_fin']) ? $_POST['fecha_fin'] : '';

// Construir la consulta SQL de acuerdo a los filtros
$whereConditions = [];
$params = [];

// Agregar condición de búsqueda si existe
$search = isset($_POST['search']) ? $_POST['search'] : '';
if ($search) {
    $whereConditions[] = 'c.nombre LIKE :search';
    $params[':search'] = "%$search%";
}

// Agregar condición de fechas si existen
if ($fechaInicio && $fechaFin) {
    $whereConditions[] = 'p.fecha_pago BETWEEN :fechaInicio AND :fechaFin';
    $params[':fechaInicio'] = $fechaInicio;
    $params[':fechaFin'] = $fechaFin;
} elseif ($fechaInicio) {
    $whereConditions[] = 'p.fecha_pago >= :fechaInicio';
    $params[':fechaInicio'] = $fechaInicio;
} elseif ($fechaFin) {
    $whereConditions[] = 'p.fecha_pago <= :fechaFin';
    $params[':fechaFin'] = $fechaFin;
}

// Crear la consulta SQL con las condiciones
$sql = "SELECT p.*, c.nombre AS nombre_cliente, c.apellido AS apellido_cliente, u.nombre_usuario AS nombre_usuario 
        FROM pagos p 
        JOIN clientes c ON p.id_cliente = c.id_cliente 
        JOIN usuarios u ON p.id_usuario = u.id_usuario";

if (!empty($whereConditions)) {
    $sql .= ' WHERE ' . implode(' AND ', $whereConditions);
}

// Añadir la cláusula ORDER BY para ordenar por ID de mayor a menor
$sql .= " ORDER BY p.id_pago DESC";  // Ordenar por id_pago en orden descendente

$sql .= " LIMIT :inicio, :registrosPorPagina";
$stmt = $pdo->prepare($sql);
foreach ($params as $key => $value) {
    $stmt->bindValue($key, $value);
}
$stmt->bindValue(':inicio', (int)$inicio, PDO::PARAM_INT);
$stmt->bindValue(':registrosPorPagina', (int)$registrosPorPagina, PDO::PARAM_INT);
$stmt->execute();
$pagos = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Función para obtener el número total de pagos
function getTotalPagos($pdo, $whereConditions, $params) {
    $sql = "SELECT COUNT(*) 
            FROM pagos p 
            JOIN clientes c ON p.id_cliente = c.id_cliente";

    if (!empty($whereConditions)) {
        $sql .= ' WHERE ' . implode(' AND ', $whereConditions);
    }
    
    $stmt = $pdo->prepare($sql);
    foreach ($params as $key => $value) {
        $stmt->bindValue($key, $value);
    }
    $stmt->execute();
    return $stmt->fetchColumn();
}

// Obtener el número total de pagos
$totalPagos = getTotalPagos($pdo, $whereConditions, $params);
$totalPaginas = ceil($totalPagos / $registrosPorPagina);
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>CRUD de Pagos - Renta de Autos</title>
    <link href="https://maxcdn.bootstrapcdn.com/bootstrap/4.5.2/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css" rel="stylesheet">
    <style>
        .module-header {
            margin-top: 30px;
            margin-bottom: 15px;
        }

        .btn-primary {
            border-radius: 50px;
            font-size: 0.875rem;
        }

        .input-group {
            width: 100%;
            max-width: 500px;
        }

        .form-control {
            border-radius: 50px 0 0 50px;
            box-shadow: none;
        }

        .input-group-append .btn-secondary {
            border-radius: 0 50px 50px 0;
        }

        .btn-info {
            border-radius: 50px;
            font-size: 0.875rem;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
        }
        #message {
            position: fixed;
            top: 10px;
            right: 10px;
            width: 300px;
            z-index: 1050;
        }
        .modal-content {
            padding: 20px;
        }
    </style>
</head>
<body>
    <?php include '../menu.php'; ?>

    <div class="main-content">
        <div class="d-flex align-items-center justify-content-center p-3 my-3 text-white bg-dark rounded shadow-sm">
    <div class="lh-1 text-center">
        <h2 class="mb-4 text-center">
            <i class="fas fa-credit-card"></i> <!-- Icono de FontAwesome -->
            Administrar Pagos
        </h2>
    </div>
</div>


<div class="d-flex flex-wrap align-items-center mb-3">
        <!-- Formulario de búsqueda -->
<form id="searchForm" class="row w-100">
    <div class="form-group col-md-4 mb-2">
        <label for="searchInput">Buscar:</label>
        <div class="input-group">
            <input type="text" class="form-control" id="searchInput" name="search" placeholder="Buscar pagos..." value="<?php echo htmlspecialchars($search); ?>">
            <div class="input-group-append">
                <button class="btn btn-secondary" type="button" id="searchButton">
                    <i class="fas fa-search"></i>
                </button>
            </div>
        </div>
    </div>

    <div class="form-group col-md-2 mb-2">
        <label for="fechaInicio">Fecha Inicio:</label>
        <input type="date" class="form-control" id="fechaInicio" name="fecha_inicio" value="<?php echo htmlspecialchars($fechaInicio); ?>">
    </div>

    <div class="form-group col-md-3 mb-2">
        <label for="fechaFin">Fecha Fin:</label>
        <div class="input-group">
            <input type="date" class="form-control" id="fechaFin" name="fecha_fin" value="<?php echo htmlspecialchars($fechaFin); ?>">
            <div class="input-group-append">
                <button type="submit" class="btn btn-secondary">
                    <i class="fas fa-filter"></i> Filtrar
                </button>
            </div>
        </div>
    </div>

    <div class="form-group col-md-3 mb-2 d-flex align-items-end">
        <a id="generatePdf" class="btn btn-info w-100" target="_blank">
            <i class="fas fa-file-pdf"></i> Generar PDF
        </a>
    </div>
</form>
</div>




            <script>
            document.getElementById('generatePdf').addEventListener('click', function() {
                var search = encodeURIComponent(document.getElementById('searchInput').value);
                var fechaInicio = encodeURIComponent(document.getElementById('fechaInicio').value);
                var fechaFin = encodeURIComponent(document.getElementById('fechaFin').value);
                
                var pdfLink = '../reportes/pagos_pdf.php?search=' + search + '&fecha_inicio=' + fechaInicio + '&fecha_fin=' + fechaFin;
                document.getElementById('generatePdf').setAttribute('href', pdfLink);
            });
            </script>

            <!-- Tabla de pagos -->
            <table class="table table-bordered table-striped">
                <thead class="thead-dark">
                    <tr>
                        <th>ID</th>
                        <th>Cliente</th>
                        <th>Monto</th>
                        <th>Fecha de Pago</th>
                        <th>Método de Pago</th>
                        <th>Usuario</th>
                        <th>Estado</th>
                        <th>Acciones</th>
                    </tr>
                </thead>
                <tbody id="paymentTableBody">
                    <?php foreach ($pagos as $pago): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($pago['id_pago']); ?></td>
                            <td><?php echo htmlspecialchars($pago['nombre_cliente']) . ' ' . htmlspecialchars($pago['apellido_cliente']); ?></td>
                            <td><?php echo htmlspecialchars(formatMonto($moneda, $pago['monto'])); ?></td>
                            <td><?php echo htmlspecialchars($pago['fecha_pago']); ?></td>
                            <td><?php echo htmlspecialchars($pago['metodo_pago']); ?></td>
                            <td><?php echo htmlspecialchars($pago['nombre_usuario']); ?></td>
                            <td><?php echo htmlspecialchars($pago['estado']); ?></td>
                            <td>
                                <!-- Agrupación de botones en una fila -->
                            <div class="btn-group" role="group" aria-label="Botones de acción">
                            <?php if ($permisos['editar']) : ?>
                                <button class="btn btn-warning btn-sm edit-btn" data-id="<?php echo $pago['id_pago']; ?>" data-toggle="modal" data-target="#editModal">
                                    <i class="fas fa-edit"></i>
                                </button>
                                <?php endif; ?>
                                <?php if ($permisos['eliminar']) : ?>
                                <button class="btn btn-danger btn-sm delete-btn" data-id="<?php echo $pago['id_pago']; ?>">
                                    <i class="fas fa-trash-alt"></i>
                                </button>
                                <?php endif; ?>
                                <?php if ($permisos['ver']) : ?>
                                <button class="btn btn-success btn-sm reprint-btn" data-id="<?php echo $pago['id_pago']; ?>">
                                    <i class="fas fa-print"></i>
                                </button>
                                <?php endif; ?>
                            </div>

                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>

            <!-- Paginación -->
            <nav aria-label="Page navigation">
                <ul class="pagination">
                    <li class="page-item <?php echo $paginaActual <= 1 ? 'disabled' : ''; ?>">
                        <a class="page-link" href="?pagina=<?php echo max(1, $paginaActual - 1); ?>" aria-label="Previous">
                            <span aria-hidden="true">&laquo;</span>
                        </a>
                    </li>
                    <?php for ($i = 1; $i <= $totalPaginas; $i++): ?>
                        <li class="page-item <?php echo $i == $paginaActual ? 'active' : ''; ?>">
                            <a class="page-link" href="?pagina=<?php echo $i; ?>"><?php echo $i; ?></a>
                        </li>
                    <?php endfor; ?>
                    <li class="page-item <?php echo $paginaActual >= $totalPaginas ? 'disabled' : ''; ?>">
                        <a class="page-link" href="?pagina=<?php echo min($totalPaginas, $paginaActual + 1); ?>" aria-label="Next">
                            <span aria-hidden="true">&raquo;</span>
                        </a>
                    </li>
                </ul>
            </nav>
    </div>

   
        <!-- Modal de edición -->
<div class="modal fade" id="editModal" tabindex="-1" role="dialog" aria-labelledby="editModalLabel" aria-hidden="true">
    <div class="modal-dialog" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="editModalLabel">Editar Pago</h5>
                <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
            <div class="modal-body">
                <form id="editForm">
                    <input type="hidden" name="id_pago" id="editId">
                    <div class="form-group">
                        <label for="editClienteNombre">Cliente</label>
                        <input type="text" class="form-control" id="editClienteNombre" name="cliente" readonly>
                    </div>
                    <div class="form-group">
                        <label for="editUsuarioNombre">Usuario</label>
                        <input type="text" class="form-control" id="editUsuarioNombre" name="usuario" readonly>
                    </div>
                    <div class="form-row">
    <div class="form-group col-md-6">
        <label for="editMonto">Monto</label>
        <input type="number" step="0.01" class="form-control" id="editMonto" name="monto" required>
    </div>
    <div class="form-group col-md-6">
        <label for="editFechaPago">Fecha de Pago</label>
        <input type="datetime-local" class="form-control" id="editFechaPago" name="fecha_pago" required>
    </div>
</div>
<div class="form-row">
    <div class="form-group col-md-6">
        <label for="editMetodoPago">Método de Pago</label>
        <select class="form-control" id="editMetodoPago" name="metodo_pago" required>
            <option value="Efectivo">Efectivo</option>
            <option value="Tarjeta de Crédito">Tarjeta de Crédito</option>
            <option value="Transferencia Bancaria">Transferencia Bancaria</option>
            <option value="Otro">Otro</option>
        </select>
    </div>
    <div class="form-group col-md-6">
        <label for="editEstado">Estado</label>
        <select class="form-control" id="editEstado" name="estado" required>
            <option value="Pendiente">Pendiente</option>
            <option value="Completo">Completo</option>
        </select>
    </div>
</div>

                    <button type="submit" class="btn btn-primary">Guardar cambios</button>
                </form>
            </div>
        </div>
    </div>
</div>
    <script src="https://code.jquery.com/jquery-3.5.1.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/popper.js@1.16.0/dist/umd/popper.min.js"></script>
    <script src="https://stackpath.bootstrapcdn.com/bootstrap/4.5.2/js/bootstrap.min.js"></script>
    <script src="pagos.js"></script>
</body>
</html>
