<?php
session_start();
require '../config.php';
require '../functions.php';

// Verificar si la caja está aperturada
if (!isCajaAperturada($pdo)) {
    header('Location: ../caja_no_aperturada.php');
    exit;
}

// Verificar si el usuario está autenticado
if (!isset($_SESSION['usuario_id'])) {
    header('Location: ../index.php');
    exit;
}

// Verificar si el usuario tiene permiso para acceder a la página
if (!usuarioTienePermiso($pdo, $_SESSION['usuario_id'], 'editar_permisos')) {
    header('Location: ../sin_permiso.php');
    exit;
}

// Obtener roles para mostrarlos en el formulario
$roles = $pdo->query("SELECT * FROM roles")->fetchAll(PDO::FETCH_ASSOC);

// Obtener permisos organizados por módulo y acción
$query = "
    SELECT p.id_permiso, p.nombre_permiso, m.nombre_modulo
    FROM permisos p
    JOIN modulos m ON p.id_modulo = m.id_modulo
    ORDER BY m.nombre_modulo, FIELD(p.nombre_permiso, 'ver_%', 'editar_%', 'borrar_%')
";
$permisos = $pdo->query($query)->fetchAll(PDO::FETCH_ASSOC);

// Agrupar permisos por módulo
$permisos_por_modulo = [];
foreach ($permisos as $permiso) {
    $permisos_por_modulo[$permiso['nombre_modulo']][] = $permiso;
}

// Inicializar variables
$mensaje = "";
$mensaje_tipo = "";
$selected_permisos = [];
$id_rol = null;

// Obtener permisos actuales del rol seleccionado (si existe)
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $id_usuario = $_POST['id_usuario'] ?? null;
    $id_rol = $_POST['id_rol'] ?? null;

    if (!$id_usuario || !$id_rol) {
        $mensaje = "Debe seleccionar un usuario y un rol.";
        $mensaje_tipo = "danger";
    } else {
        // Eliminar permisos antiguos
        $stmt = $pdo->prepare("DELETE FROM roles_permisos WHERE id_rol = :id_rol");
        $stmt->execute([':id_rol' => $id_rol]);

        // Asignar permisos seleccionados
        if (isset($_POST['permisos'])) {
            foreach ($_POST['permisos'] as $id_permiso) {
                $stmt = $pdo->prepare("INSERT INTO roles_permisos (id_rol, id_permiso) VALUES (:id_rol, :id_permiso)");
                $stmt->execute([':id_rol' => $id_rol, ':id_permiso' => $id_permiso]);
            }
        }

        $mensaje = "Permisos actualizados correctamente.";
        $mensaje_tipo = "success";
    }
}

// Si se seleccionó un rol, obtener sus permisos actuales
if (isset($_GET['id_rol']) || $id_rol) {
    $id_rol = $_GET['id_rol'] ?? $id_rol;
    $stmt = $pdo->prepare("SELECT id_permiso FROM roles_permisos WHERE id_rol = :id_rol");
    $stmt->execute([':id_rol' => $id_rol]);
    $selected_permisos = $stmt->fetchAll(PDO::FETCH_COLUMN);
}

// Obtener usuarios que están asignados a algún rol
$usuarios_query = "
    SELECT u.id_usuario, u.nombre
    FROM usuarios u
    JOIN usuarios_roles ur ON u.id_usuario = ur.id_usuario
    GROUP BY u.id_usuario, u.nombre
";
$usuarios = $pdo->query($usuarios_query)->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Editar Roles y Permisos</title>
    <link rel="shortcut icon" href="../img/favi.ico">
    <link href="https://maxcdn.bootstrapcdn.com/bootstrap/4.5.2/css/bootstrap.min.css" rel="stylesheet">
    <style>
        .module-header {
            margin-top: 30px;
            margin-bottom: 15px;
        }
        .permission-card {
            margin-bottom: 10px;
            padding: 10px;
            border: 1px solid #ddd;
            border-radius: 4px;
            box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
        }
        .permission-card input[type="checkbox"] {
            margin-right: 10px;
        }
        .card-group {
            margin-bottom: 20px;
        }
        .form-row {
            margin-bottom: 15px;
        }
        .select-all {
            margin-bottom: 15px;
        }
    </style>
</head>
<body>
    <?php include '../menu.php'; ?>

    <div class="container mt-4">
    <div class="d-flex align-items-center justify-content-center p-3 my-3 text-white bg-info rounded shadow-sm">
    <div class="lh-1 text-center">
        <h2 class="mb-4 text-center">
            <i class="fas fa-edit"></i> <!-- Icono de edición de FontAwesome -->
            Editar Roles y Permisos
        </h2>
    </div>
</div>

        <!-- Mostrar mensaje -->
        <?php if (!empty($mensaje)): ?>
            <div class="alert alert-<?php echo $mensaje_tipo; ?>" role="alert">
                <?php echo $mensaje; ?>
            </div>
        <?php endif; ?>

        <form method="post">
            <div class="form-row">
                <div class="form-group col-md-2">
                    <label for="id_usuario">Usuario:</label>
                    <select id="id_usuario" name="id_usuario" class="form-control" required>
                        <option value="" disabled selected>Seleccionar usuario</option>
                        <?php foreach ($usuarios as $usuario) : ?>
                            <option value="<?= htmlspecialchars($usuario['id_usuario']) ?>"><?= htmlspecialchars($usuario['nombre']) ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <div class="form-group col-md-2">
                    <label for="id_rol">Rol:</label>
                    <select id="id_rol" name="id_rol" class="form-control" required>
                        <option value="" disabled selected>Seleccionar Rol</option>
                        <?php foreach ($roles as $rol) : ?>
                            <option value="<?= htmlspecialchars($rol['id_rol']) ?>" <?= isset($id_rol) && $id_rol == $rol['id_rol'] ? 'selected' : '' ?>><?= htmlspecialchars($rol['nombre_rol']) ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="form-group col-md-4 d-flex align-items-end">
                    <a href="asignar_permisos.php" class="btn btn-info mr-2">
                        <i class="fas fa-users-cog"></i> Gestionar Permisos
                    </a>
                </div>
            </div>

            <div class="form-group select-all">
                <input type="checkbox" id="select_all" class="mr-2">
                <label for="select_all">Seleccionar todos</label>
            </div>

            <div class="form-group">
    <table class="table table-bordered table-striped">
        <thead class="thead-dark">
            <tr>
                <th>Módulo</th>
                <th>Ver</th>
                <th>Editar</th>
                <th>Borrar</th>
            </tr>
        </thead>
        <tbody>
            <?php foreach ($permisos_por_modulo as $modulo => $permisos) : ?>
                <tr>
                    <td><?= htmlspecialchars($modulo) ?></td>
                    <td>
                        <?php foreach ($permisos as $permiso) : ?>
                            <?php if (strpos($permiso['nombre_permiso'], 'ver_') === 0) : ?>
                                <div class="form-check">
                                    <input type="checkbox" id="permiso_<?= htmlspecialchars($permiso['id_permiso']) ?>" name="permisos[]" value="<?= htmlspecialchars($permiso['id_permiso']) ?>" class="form-check-input" <?= in_array($permiso['id_permiso'], $selected_permisos) ? 'checked' : '' ?>>
                                    <label class="form-check-label" for="permiso_<?= htmlspecialchars($permiso['id_permiso']) ?>"></label>
                                </div>
                            <?php endif; ?>
                        <?php endforeach; ?>
                    </td>
                    <td>
                        <?php foreach ($permisos as $permiso) : ?>
                            <?php if (strpos($permiso['nombre_permiso'], 'editar_') === 0) : ?>
                                <div class="form-check">
                                    <input type="checkbox" id="permiso_<?= htmlspecialchars($permiso['id_permiso']) ?>" name="permisos[]" value="<?= htmlspecialchars($permiso['id_permiso']) ?>" class="form-check-input" <?= in_array($permiso['id_permiso'], $selected_permisos) ? 'checked' : '' ?>>
                                    <label class="form-check-label" for="permiso_<?= htmlspecialchars($permiso['id_permiso']) ?>"></label>
                                </div>
                            <?php endif; ?>
                        <?php endforeach; ?>
                    </td>
                    <td>
                        <?php foreach ($permisos as $permiso) : ?>
                            <?php if (strpos($permiso['nombre_permiso'], 'borrar_') === 0) : ?>
                                <div class="form-check">
                                    <input type="checkbox" id="permiso_<?= htmlspecialchars($permiso['id_permiso']) ?>" name="permisos[]" value="<?= htmlspecialchars($permiso['id_permiso']) ?>" class="form-check-input" <?= in_array($permiso['id_permiso'], $selected_permisos) ? 'checked' : '' ?>>
                                    <label class="form-check-label" for="permiso_<?= htmlspecialchars($permiso['id_permiso']) ?>"></label>
                                </div>
                            <?php endif; ?>
                        <?php endforeach; ?>
                    </td>
                </tr>
            <?php endforeach; ?>
        </tbody>
    </table>
</div>


            <button type="submit" class="btn btn-info">Guardar Cambios</button>
        </form>
    </div>

    <script src="https://code.jquery.com/jquery-3.5.1.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/@popperjs/core@2.5.4/dist/umd/popper.min.js"></script>
    <script src="https://maxcdn.bootstrapcdn.com/bootstrap/4.5.2/js/bootstrap.min.js"></script>
    <script>
        $(document).ready(function() {
            $('#select_all').click(function() {
                $('input[type="checkbox"]').prop('checked', this.checked);
            });
                // Manejar la selección de un usuario
            $('#id_usuario').change(function() {
                var id_usuario = $(this).val();

                // Realizar la solicitud AJAX para obtener los permisos del usuario
                $.ajax({
                    url: 'obtener_permisos_usuario.php',
                    method: 'POST',
                    data: { id_usuario: id_usuario },
                    dataType: 'json',
                    success: function(response) {
                        // Desactivar todos los checkboxes
                        $('input[type="checkbox"]').prop('checked', false);

                        // Activar los checkboxes que corresponden a los permisos del usuario
                        $.each(response, function(index, id_permiso) {
                            $('#permiso_' + id_permiso).prop('checked', true);
                        });
                    },
                    error: function() {
                        alert('Error al obtener los permisos del usuario.');
                    }
                });
            });

            // Seleccionar/Deseleccionar todos los checkboxes
            $('#select_all').click(function() {
                $('input[type="checkbox"]').prop('checked', this.checked);
            });
        });
    
    </script>
</body>
</html>
