<?php
session_start();
require '../config.php';
require '../functions.php';

// Verificar si la caja está aperturada
if (!isCajaAperturada($pdo)) {
    header('Location: ../caja_no_aperturada.php');
    exit;
}

// Verifica si el usuario está autenticado
if (!isset($_SESSION['usuario_id'])) {
    header('Location: ../index.php');
    exit;
}

// Verifica si el usuario tiene permiso para acceder a la página
if (!usuarioTienePermiso($pdo, $_SESSION['usuario_id'], 'ver_permisos')) {
    header('Location: ../sin_permiso.php');
    exit;
}

// Verifica los permisos del usuario para ver, editar y eliminar
$usuario_id = $_SESSION['usuario_id'];
$permisos = [
    'ver' => usuarioTienePermiso($pdo, $usuario_id, 'ver_permisos'),
    'editar' => usuarioTienePermiso($pdo, $usuario_id, 'editar_permisos'),
    'eliminar' => usuarioTienePermiso($pdo, $usuario_id, 'borrar_permisos'),
];

// Manejo del formulario para agregar un nuevo rol
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['agregar_rol'])) {
    $nombre_rol = trim($_POST['nombre_rol']);
    
    if (!empty($nombre_rol)) {
        $stmt = $pdo->prepare("INSERT INTO roles (nombre_rol) VALUES (:nombre_rol)");
        $stmt->execute([':nombre_rol' => $nombre_rol]);
        $mensaje = "Rol agregado correctamente.";
    } else {
        $mensaje = "El nombre del rol no puede estar vacío.";
    }
}

// Manejo de la modificación de un rol
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['modificar_rol'])) {
    $id_rol = $_POST['id_rol'];
    $nombre_rol = trim($_POST['nombre_rol']);

    if (!empty($nombre_rol)) {
        $stmt = $pdo->prepare("UPDATE roles SET nombre_rol = :nombre_rol WHERE id_rol = :id_rol");
        $stmt->execute([':nombre_rol' => $nombre_rol, ':id_rol' => $id_rol]);
        $mensaje = "Rol modificado correctamente.";
    } else {
        $mensaje = "El nombre del rol no puede estar vacío.";
    }
}

// Obtener la lista de roles
$roles = $pdo->query("SELECT * FROM roles")->fetchAll(PDO::FETCH_ASSOC);

// Manejo de la eliminación de un rol
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['eliminar_rol'])) {
    $id_rol = $_POST['id_rol'];

    $stmt = $pdo->prepare("DELETE FROM roles WHERE id_rol = :id_rol");
    $stmt->execute([':id_rol' => $id_rol]);
    $mensaje = "Rol eliminado correctamente.";
}
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gestión de Roles</title>
    <link rel="shortcut icon" href="../img/favi.ico">
    <link href="https://maxcdn.bootstrapcdn.com/bootstrap/4.5.2/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.3/css/all.min.css">
    <style>
        .table td, .table th {
            vertical-align: middle;
        }
        .btn-group {
            display: flex;
            justify-content: center;
        }
    </style>
</head>
<body>
    <?php include '../menu.php'; ?>

    <div class="container mt-4">
    <div class="d-flex align-items-center justify-content-center p-3 my-3 text-white bg-info rounded shadow-sm">
    <div class="lh-1 text-center">
        <h2 class="mb-4 text-center">
            <i class="fas fa-users-cog"></i> <!-- Icono de FontAwesome -->
            Gestión de Roles
        </h2>
    </div>
</div>

        <?php if (isset($mensaje)): ?>
            <div class="alert alert-info text-center"><?= $mensaje ?></div>
        <?php endif; ?>

        <div class="card mb-4">
            <div class="card-header bg-info text-white">
                <i class="fas fa-plus"></i> Agregar Nuevo Rol
            </div>
            <div class="card-body">
                <form method="post">
                    <div class="form-row align-items-end">
                        <div class="col-md-2">
                            <label for="nombre_rol">Nombre del Rol:</label>
                            <input type="text" id="nombre_rol" name="nombre_rol" class="form-control" required>
                        </div>
                        <div class="col-md-4 text-left">
                            <button type="submit" name="agregar_rol" class="btn btn-success mt-3">
                                <i class="fas fa-check"></i> Agregar Rol
                            </button>
                        </div>
                        <div class="col-md-4 text-right">
                    <a href="asignar_permisos.php" class="btn btn-info mr-2">
                        <i class="fas fa-users-cog"></i> Gestionar Permisos
                    </a>
                </div>
                    </div>
                </form>
            </div>
        </div>

        <table class="table table-bordered table-striped">
            <thead class="thead-dark">
                <tr>
                    <th>ID</th>
                    <th>Nombre del Rol</th>
                    
                </tr>
            </thead>
            <tbody>
                <?php foreach ($roles as $rol): ?>
                    <tr>
                        <td><?= $rol['id_rol'] ?></td>
                        <td>
                            <form method="post" class="d-flex align-items-center">
                                <input type="hidden" name="id_rol" value="<?= $rol['id_rol'] ?>">
                                <input type="text" name="nombre_rol" value="<?= $rol['nombre_rol'] ?>" class="form-control mr-3" required>
                                <div class="btn-group">
                                <?php if ($permisos['editar']) : ?>
                                    <button type="submit" name="modificar_rol" class="btn btn-info btn-sm mr-2">
                                        <i class="fas fa-edit"></i> Modificar
                                    </button>
                                    <?php endif; ?>
                                    <?php if ($permisos['eliminar']) : ?>
                                    <button type="submit" name="eliminar_rol" class="btn btn-danger btn-sm" onclick="return confirm('¿Estás seguro de eliminar este rol?')">
                                        <i class="fas fa-trash"></i> Eliminar
                                    </button>
                                    <?php endif; ?>
                                </div>
                            </form>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>

    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://maxcdn.bootstrapcdn.com/bootstrap/4.5.2/js/bootstrap.min.js"></script>
</body>
</html>
