<?php
session_start();
require '../config.php';
require '../functions.php';

// Verificar si la caja está aperturada
if (!isCajaAperturada($pdo)) {
    header('Location: ../caja_no_aperturada.php');
    exit;
}

// Verifica si el usuario está autenticado
if (!isset($_SESSION['usuario_id'])) {
    header('Location: ../index.php');
    exit;
}
// Verifica si el usuario tiene permiso para acceder a la página
if (!usuarioTienePermiso($pdo, $_SESSION['usuario_id'], 'ver_rentas')) {
    header('Location: ../sin_permiso.php');
    exit;
}

// Verifica los permisos del usuario para ver, editar y eliminar
$usuario_id = $_SESSION['usuario_id'];
$permisos = [
    'ver' => usuarioTienePermiso($pdo, $usuario_id, 'ver_rentas'),
    'editar' => usuarioTienePermiso($pdo, $usuario_id, 'editar_rentas'),
    'eliminar' => usuarioTienePermiso($pdo, $usuario_id, 'borrar_rentas'),
];


// Número de registros por página
$registrosPorPagina = 10;

// Obtener el número de página actual
$paginaActual = isset($_GET['pagina']) ? (int)$_GET['pagina'] : 1;
$paginaActual = $paginaActual > 0 ? $paginaActual : 1;

// Calcular el desplazamiento
$inicio = ($paginaActual - 1) * $registrosPorPagina;

// Función para obtener las rentas con paginación
function getRentas($pdo, $inicio, $registrosPorPagina, $search = '') {
    if ($search === 'fecha_devolucion_hoy') {
        $fechaActual = date('Y-m-d');
        $sql = "
        SELECT rentas.id_renta, clientes.nombre AS nombre_cliente, clientes.apellido AS apellido_cliente, 
               CONCAT(habitaciones.nombre_habitacion, ' ', habitaciones.numero_habitacion	, ' ', habitaciones.tipo) AS auto, usuarios.nombre_usuario, 
               rentas.fecha_renta, rentas.fecha_devolucion, rentas.precio_total, rentas.estado, rentas.pago,
               DATEDIFF(rentas.fecha_devolucion, rentas.fecha_renta) AS dias_renta
        FROM rentas
        JOIN clientes ON rentas.id_cliente = clientes.id_cliente
        JOIN habitaciones ON rentas.id_auto = habitaciones.id_habitacion
        JOIN usuarios ON rentas.id_usuario = usuarios.id_usuario
        WHERE rentas.fecha_devolucion = :fecha_actual
        ORDER BY rentas.id_renta DESC
        LIMIT :inicio, :registrosPorPagina";
        $stmt = $pdo->prepare($sql);
        $stmt->bindValue(':fecha_actual', $fechaActual);
    } else {
        $sql = "
        SELECT rentas.id_renta, clientes.nombre AS nombre_cliente, clientes.apellido AS apellido_cliente, 
               CONCAT(habitaciones.nombre_habitacion, ' ', habitaciones.numero_habitacion	, ' ', habitaciones.tipo) AS auto, usuarios.nombre_usuario, 
               rentas.fecha_renta, rentas.fecha_devolucion, rentas.precio_total, rentas.estado, rentas.pago,
               DATEDIFF(rentas.fecha_devolucion, rentas.fecha_renta) AS dias_renta
        FROM rentas
        JOIN clientes ON rentas.id_cliente = clientes.id_cliente
        JOIN habitaciones ON rentas.id_auto = habitaciones.id_habitacion
        JOIN usuarios ON rentas.id_usuario = usuarios.id_usuario
        WHERE CONCAT(clientes.nombre, ' ', clientes.apellido, ' ', habitaciones.nombre_habitacion, ' ', habitaciones.numero_habitacion	, ' ', usuarios.nombre_usuario, ' ', rentas.fecha_renta, ' ', rentas.fecha_devolucion, ' ', rentas.precio_total, ' ', rentas.estado) LIKE :search
        ORDER BY rentas.id_renta DESC
        LIMIT :inicio, :registrosPorPagina";
        $stmt = $pdo->prepare($sql);
        $stmt->bindValue(':search', "%$search%");
    }
    $stmt->bindValue(':inicio', (int)$inicio, PDO::PARAM_INT);
    $stmt->bindValue(':registrosPorPagina', (int)$registrosPorPagina, PDO::PARAM_INT);
    $stmt->execute();
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}


// Función para obtener el número total de rentas
function getTotalRentas($pdo, $search = '') {
    if ($search === 'fecha_devolucion_hoy') {
        $fechaActual = date('Y-m-d');
        $sql = "
        SELECT COUNT(*) 
        FROM rentas
        JOIN clientes ON rentas.id_cliente = clientes.id_cliente
        JOIN habitaciones ON rentas.id_auto = habitaciones.id_habitacion
        JOIN usuarios ON rentas.id_usuario = usuarios.id_usuario
        WHERE rentas.fecha_devolucion = :fecha_actual";
        $stmt = $pdo->prepare($sql);
        $stmt->bindValue(':fecha_actual', $fechaActual);
    } else {
        $sql = "
        SELECT COUNT(*) 
        FROM rentas
        JOIN clientes ON rentas.id_cliente = clientes.id_cliente
        JOIN habitaciones ON rentas.id_auto = habitaciones.id_habitacion
        JOIN usuarios ON rentas.id_usuario = usuarios.id_usuario
        WHERE CONCAT(clientes.nombre, ' ', clientes.apellido, ' ', habitaciones.nombre_habitacion, ' ', habitaciones.numero_habitacion	, ' ', usuarios.nombre_usuario, ' ', rentas.fecha_renta, ' ', rentas.fecha_devolucion, ' ', rentas.precio_total, ' ', rentas.estado) LIKE :search";
        $stmt = $pdo->prepare($sql);
        $stmt->bindValue(':search', "%$search%");
    }
    $stmt->execute();
    return $stmt->fetchColumn();
}


// Buscar rentas
$search = isset($_POST['searchR']) ? $_POST['searchR'] : '';
$totalRentas = getTotalRentas($pdo, $search);
$totalPaginas = ceil($totalRentas / $registrosPorPagina);
$rentas = getRentas($pdo, $inicio, $registrosPorPagina, $search);

// Obtener la fecha actual
$fechaActual = date('Y-m-d');
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Reservaciones - Sistema Hotel</title>
    <link href="https://maxcdn.bootstrapcdn.com/bootstrap/4.5.2/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css" rel="stylesheet">
    <style>
         .module-header {
            margin-top: 30px;
            margin-bottom: 15px;
        }
        .btn-primary {
            border-radius: 50px;
            font-size: 0.875rem;
        }

        .input-group {
            width: 100%;
            max-width: 500px;
        }

        .form-control {
            border-radius: 50px 0 0 50px;
            box-shadow: none;
        }

        .input-group-append .btn-secondary {
            border-radius: 0 50px 50px 0;
        }

        .btn-info {
            border-radius: 50px;
            font-size: 0.875rem;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
        }
        #message {
            position: fixed;
            top: 10px;
            right: 10px;
            width: 300px;
            z-index: 1050; /* Asegúrate de que esté encima de otros elementos */
        }
       .modal-content {
            padding: 20px;
        }
        .search-results {
            border: 1px solid #ddd;
            max-height: 200px;
            overflow-y: auto;
            position: absolute;
            width: 100%;
            background-color: white;
            z-index: 1000;
        }
        .search-results div {
            padding: 8px;
            cursor: pointer;
        }
        .search-results div:hover {
            background-color: #f1f1f1;
        }
        highlight {
            background-color: #ffcccc; /* Rojo suave */
        }
    </style>
</head>
<body>
    <?php include '../menu.php'; ?>

    <div class="main-content">
        <div class="d-flex align-items-center justify-content-center p-3 my-3 text-white bg-dark rounded shadow-sm">
    <div class="lh-1 text-center">
        <h2 class="mb-4 text-center">
            <i class="fas fa-calendar-alt"></i> <!-- Icono de FontAwesome -->
            Reservaciones
        </h2>
    </div>
</div>

<div class="d-flex flex-wrap align-items-center mb-3">
    <!-- Formulario de búsqueda -->
<form id="searchForm" class="row w-100">
    <div class="form-group col-md-4 mb-2">
        <label for="searchInputR">Buscar:</label>
        <div class="input-group">
            <input type="text" class="form-control" id="searchInputR" name="search" placeholder="Buscar...Cliente, habitaciones" value="<?php echo htmlspecialchars($search); ?>">
            <div class="input-group-append">
                <button class="btn btn-secondary" type="button" id="searchButton">
                    <i class="fas fa-search"></i>
                </button>
            </div>
        </div>
    </div>

    <div class="form-group col-md-2 mb-2">
        <label for="fechaInicio">Fecha Inicio:</label>
        <input type="date" class="form-control" id="fechaInicio" name="fecha_inicio" value="<?php echo htmlspecialchars($fechaInicio); ?>">
    </div>

    <div class="form-group col-md-3 mb-2">
        <label for="fechaFin">Fecha Fin:</label>
        <div class="input-group">
            <input type="date" class="form-control" id="fechaFin" name="fecha_fin" value="<?php echo htmlspecialchars($fechaFin); ?>">
            <div class="input-group-append">
                <button type="submit" class="btn btn-secondary">
                    <i class="fas fa-filter"></i> Filtrar
                </button>
            </div>
        </div>
    </div>

    <div class="form-group col-md-3 mb-2 d-flex align-items-end">
        <a id="generatePdf" class="btn btn-info w-100" target="_blank">
            <i class="fas fa-file-pdf"></i> Generar PDF
        </a>
    </div>
</form>

</div>
            <script>
            document.getElementById('generatePdf').addEventListener('click', function() {
                var search = encodeURIComponent(document.getElementById('searchInputR').value);
                var fechaInicio = encodeURIComponent(document.getElementById('fechaInicio').value);
                var fechaFin = encodeURIComponent(document.getElementById('fechaFin').value);
                
                var pdfLink = '../reportes/rentas_pdf.php?search=' + search + '&fecha_inicio=' + fechaInicio + '&fecha_fin=' + fechaFin;
                document.getElementById('generatePdf').setAttribute('href', pdfLink);
            });
            </script>

            <!-- Tabla de rentas -->
            <table class="table table-bordered table-striped">
                <thead class="thead-dark">
                    <tr>
                        <th>ID</th>
                        <th>Cliente</th>
                        <th>Habitación</th>
                        <th>Fecha/check-in</th>
                        <th>Fecha/check-out</th>
                        <th>Estadía</th>
                        <th>Total</th>
                        <th>Estado</th>
                        <th>Pago</th>
                        <th>Usuario</th>
                    </tr>
                </thead>
                <tbody id="rentaTableBody">
                    <?php foreach ($rentas as $renta): ?>
                        <?php 
                            // Verificar si la fecha de devolución es igual a la fecha actual
                            $highlightClass = ($renta['fecha_devolucion'] == $fechaActual) ? 'highlight' : '';
                        ?>
                        <tr class="<?php echo $highlightClass; ?>">
                            <td><?php echo htmlspecialchars($renta['id_renta']); ?></td>
                            <td><?php echo htmlspecialchars($renta['nombre_cliente'].' '.$renta['apellido_cliente']); ?></td>
                            <td><?php echo htmlspecialchars($renta['auto']); ?></td>
                            <td><?php echo htmlspecialchars(formatearFecha($renta['fecha_renta'])); ?></td>
                            <td><?php echo htmlspecialchars(formatearFecha($renta['fecha_devolucion'])); ?></td>
                            <td><span class="badge badge-primary"><?php echo htmlspecialchars($renta['dias_renta']); ?></span></td>
                            <td><?php echo htmlspecialchars(formatMonto($moneda, $renta['precio_total'])); ?></td>
                            <td><?php echo htmlspecialchars($renta['estado']); ?></td>
                            <td>
                                <?php 
                                    // Asignar la clase de badge según el estado del pago
                                    $badgeClass = $renta['pago'] === 'Pendiente' ? 'badge-danger' : 'badge-success';
                                    ?>
                                    <span class="badge <?php echo htmlspecialchars($badgeClass); ?>">
                                        <?php echo htmlspecialchars($renta['pago']); ?>
                                    </span>
                                </td>
                                <td><?php echo htmlspecialchars($renta['nombre_usuario']); ?></td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>


            <!-- Paginación -->
            <nav aria-label="Page navigation">
                <ul class="pagination">
                    <li class="page-item <?php echo $paginaActual <= 1 ? 'disabled' : ''; ?>">
                        <a class="page-link" href="?pagina=<?php echo max(1, $paginaActual - 1); ?>" aria-label="Previous">
                            <span aria-hidden="true">&laquo;</span>
                        </a>
                    </li>
                    <?php for ($i = 1; $i <= $totalPaginas; $i++): ?>
                        <li class="page-item <?php echo $i == $paginaActual ? 'active' : ''; ?>">
                            <a class="page-link" href="?pagina=<?php echo $i; ?>"><?php echo $i; ?></a>
                        </li>
                    <?php endfor; ?>
                    <li class="page-item <?php echo $paginaActual >= $totalPaginas ? 'disabled' : ''; ?>">
                        <a class="page-link" href="?pagina=<?php echo min($totalPaginas, $paginaActual + 1); ?>" aria-label="Next">
                            <span aria-hidden="true">&raquo;</span>
                        </a>
                    </li>
                </ul>
            </nav>

        <!-- Contenedor para mensajes -->
        <div id="message" class="alert d-none"></div>

<!-- Modal para agregar una nueva renta -->
<div class="modal fade" id="addModal" tabindex="-1" role="dialog" aria-labelledby="addModalLabel" aria-hidden="true">
    <div class="modal-dialog" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="addModalLabel">Agregar Nueva Renta</h5>
                <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
            <div class="modal-body">
                <form id="addForm">
                <div class="form-group">
        <label for="addClienteInput">Cliente</label>
        <input type="text" class="form-control" id="addClienteInput" placeholder="Buscar cliente..." autocomplete="off">
        <div id="addClienteResults" class="search-results"></div>
        <input type="hidden" id="addClienteId" name="id_cliente"> <!-- Campo oculto para el ID del cliente -->
    </div>
    <div class="form-group">
        <label for="addAutoInput">Auto</label>
        <input type="text" class="form-control" id="addAutoInput" placeholder="Buscar auto..." autocomplete="off">
        <div id="addAutoResults" class="search-results"></div>
        <input type="hidden" id="addAutoId" name="id_auto"> <!-- Campo oculto para el ID del auto -->
    </div>
    <div class="form-row">
    <div class="form-group col-md-6">
        <label for="addFechaRenta">Fecha Renta</label>
        <input type="date" class="form-control" id="addFechaRenta" name="fecha_renta" required>
    </div>
    <div class="form-group col-md-6">
        <label for="addFechaDevolucion">Fecha Devolución</label>
        <input type="date" class="form-control" id="addFechaDevolucion" name="fecha_devolucion" required>
    </div>
</div>
<div class="form-row">
    <div class="form-group col-md-6">
        <label for="addDias">Número de Días</label>
        <input type="text" class="form-control" id="addDias" name="dias" readonly>
    </div>
    <div class="form-group col-md-6">
        <label for="addPrecioTotal">Precio Total</label>
        <input type="number" class="form-control" id="addPrecioTotal" name="precio_total" step="0.01" required autocomplete="off">
    </div>
</div>

                    <div class="form-group">
                        <label for="addEstado">Estado</label>
                        <select class="form-control" id="addEstado" name="estado" required autocomplete="off">
                            <option value="Reservado">Reservado</option>
                            <option value="Entregado">Entregado</option>
                            <option value="Devuelto">Devuelto</option>
                        </select>
                    </div>
                    <button type="submit" class="btn btn-primary">Agregar Renta</button>
                </form>
            </div>
        </div>
    </div>
</div>

<!-- Modal de edición -->
<div class="modal fade" id="editModal" tabindex="-1" role="dialog" aria-labelledby="editModalLabel" aria-hidden="true">
    <div class="modal-dialog" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="editModalLabel">Editar Renta</h5>
                <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
            <div class="modal-body">
                <form id="editForm">
                <input type="hidden" name="id_renta" id="editId">
                <div class="form-group">
        <label for="editClienteInput">Cliente</label>
        <input type="text" class="form-control" id="editClienteInput" placeholder="Buscar cliente..." autocomplete="off">
        <div id="editClienteResults" class="search-results"></div>
        <input type="hidden" id="editClienteId" name="id_cliente"> <!-- Campo oculto para el ID del cliente -->
    </div>
    <div class="form-group">
        <label for="editAutoInput">Auto</label>
        <input type="text" class="form-control" id="editAutoInput" placeholder="Buscar auto..." autocomplete="off">
        <div id="editAutoResults" class="search-results"></div>
        <input type="hidden" id="editAutoId" name="id_auto"> <!-- Campo oculto para el ID del auto -->
    </div>
    <div class="form-row">
    <div class="form-group col-md-6">
        <label for="editFechaRenta">Fecha Renta</label>
        <input type="date" class="form-control" id="editFechaRenta" name="fecha_renta" required>
    </div>
    <div class="form-group col-md-6">
        <label for="editFechaDevolucion">Fecha Devolución</label>
        <input type="date" class="form-control" id="editFechaDevolucion" name="fecha_devolucion" required>
    </div>
</div>

<div class="form-row">
    <div class="form-group col-md-6">
        <label for="editDias">Número de Días</label>
        <input type="text" class="form-control" id="editDias" name="dias" readonly>
    </div>
    <div class="form-group col-md-6">
        <label for="editPrecioTotal">Precio Total</label>
        <input type="number" class="form-control" id="editPrecioTotal" name="precio_total" step="0.01" required autocomplete="off">
    </div>
</div>

                    <div class="form-group">
                        <label for="editEstado">Estado</label>
                        <select class="form-control" id="editEstado" name="estado" required autocomplete="off">
                            <option value="Reservado">Reservado</option>
                            <option value="Entregado">Entregado</option>
                            <option value="Devuelto">Devuelto</option>
                        </select>
                    </div>
                    <button type="submit" class="btn btn-primary">Guardar cambios</button>
                </form>
            </div>
        </div>
    </div>
</div>

<!-- Modal para detalles de la reservación -->
<div class="modal fade" id="detailsModal" tabindex="-1" role="dialog" aria-labelledby="detailsModalLabel" aria-hidden="true">
    <div class="modal-dialog" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="detailsModalLabel">Detalles de la Reservación</h5>
                <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
            <div class="modal-body">
                <div id="reservationDetails"></div>
                <form id="paymentForm">
                    <input type="hidden" id="id_cliente">
                    <input type="hidden" id="id_renta">
                    <div class="form-group">
                        <label for="amount">Monto a Pagar:</label>
                        <input type="number" step="0.01" class="form-control" id="amount" required>
                    </div>
                    <div class="form-group">
                        <label for="metodo_pago">Método de Pago:</label>
                        <select class="form-control" id="metodo_pago" required>
                            <option value="Efectivo">Efectivo</option>
                            <option value="Tarjeta de Crédito">Tarjeta de Crédito</option>
                            <option value="Transferencia Bancaria">Transferencia Bancaria</option>
                            <option value="Otro">Otro</option>
                        </select>
                    </div>
                    <button type="submit" class="btn btn-primary">Realizar Pago</button>
                </form>
            </div>
        </div>
    </div>
</div>




    </div>

    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/@popperjs/core@2.11.6/dist/umd/popper.min.js"></script>
    <script src="https://maxcdn.bootstrapcdn.com/bootstrap/4.5.2/js/bootstrap.min.js"></script>
    <script src="rentas.js"></script>
    <script> 

$(document).ready(function() {
    $('#filterButton').click(function() {
        var fechaDevolucion = $('#fechaDevolucionInput').val();
        $.ajax({
            url: 'filtrar_rentas.php',
            type: 'POST',
            data: { fecha_devolucion: fechaDevolucion },
            success: function(response) {
                $('#rentaTableBody').html(response);
            }
        });
    });
});

    </script>


</body>
</html>


</body>
</html>
